#pragma once

#include "tcp_connector.h"
#include "timeout_map.h"

#include <yandex_io/libs/ipc/i_connector.h>
#include <yandex_io/libs/ipc/i_server.h>

#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/libs/threading/lifetime.h>

#include <yandex_io/protos/quasar_proto.pb.h>

namespace quasar::ipc::detail::datacratic {

    class QuasarConnectorImpl: public TCPConnector {
        QuasarConnectorImpl(std::string serviceName, Lifetime::Tracker /*tracker*/, std::shared_ptr<YandexIO::Configuration> configuration, std::shared_ptr<ICallbackQueue> /*callbackQueue*/);

    public:
        using MessageHandler = IConnector::MessageHandler;
        using OnDone = IConnector::OnDone;
        using OnError = IConnector::OnError;

        QuasarConnectorImpl(std::string serviceName, std::shared_ptr<YandexIO::Configuration> configuration);
        QuasarConnectorImpl(std::string serviceName, const Lifetime& /*lifetime*/, std::shared_ptr<YandexIO::Configuration> configuration, std::shared_ptr<ICallbackQueue> /*callbackQueue*/);
        ~QuasarConnectorImpl();

        const std::string& serviceName() const;

        void setMessageHandler(MessageHandler handler);
        void setSilentMode(bool silentMode);

        bool tryConnectToService();
        void connectToService();
        void connectToTcpHost(const std::string& hostname, int port);

        // EventBus mode
        bool sendMessage(const ipc::Message& request);

        // Request-Reply mode
        void sendRequest(ipc::Message& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout);
        ipc::SharedMessage sendRequestSync(ipc::Message& message, std::chrono::milliseconds timeout);

    private:
        using HandlersTimeoutMap = TimeoutMap<std::string, std::pair<OnDone, OnError>>;
        using HandlersMap = std::unordered_map<std::string, std::pair<OnDone, OnError>>;

        void handleMessageReceived(const std::string& message) override;
        std::function<void()> doBeforeDisconnect(std::shared_ptr<TCPConnectionHandler> connection) override;
        void runErrorHandlers(HandlersMap& handlers, const std::string& errorMessage);

        template <class F, typename... Args>
        void invoke(F func, Args&&... args);

    private:
        const std::shared_ptr<YandexIO::Configuration> configuration_;
        const Lifetime::Tracker tracker_;
        const std::shared_ptr<ICallbackQueue> callbackQueue_;
        const std::string serviceName_;

        mutable std::mutex handlersMutex_;
        HandlersTimeoutMap handlers_;
        MessageHandler onQuasarMessage_;
        bool silent_ = false;
    };

} // namespace quasar::ipc::detail::datacratic
