#include "quasar_server.h"

#include "length_value_tokenizer.h"

#include <yandex_io/libs/ipc/helpers.h>

#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;
using namespace quasar::ipc::detail::datacratic;

QuasarServer::QuasarServer(std::string serviceName, std::shared_ptr<YandexIO::Configuration> configuration)
    : impl_(std::move(configuration))
    , serviceName_(std::move(helpers::checkServiceName(serviceName)))
{
}

QuasarServer::QuasarServer(std::string serviceName, std::shared_ptr<YandexIO::Configuration> configuration, const Lifetime& lifetime, std::shared_ptr<ICallbackQueue> callbackQueue)
    : impl_(lifetime, std::move(configuration), std::move(callbackQueue))
    , serviceName_(std::move(helpers::checkServiceName(serviceName)))
{
}

QuasarServer::~QuasarServer() = default;

const std::string& QuasarServer::serviceName() const {
    return serviceName_;
}

void QuasarServer::setMessageHandler(MessageHandler handler) {
    impl_.setMessageHandler([handler = std::move(handler)](const SharedMessage& message, auto* connection) {
        return handler(message, *connection);
    });
}

void QuasarServer::setClientConnectedHandler(ClientHandler handler) {
    impl_.setClientConnectedHandler([handler = std::move(handler)](auto& connection) {
        return handler(*connection);
    });
}

void QuasarServer::setClientDisconnectedHandler(ClientHandler handler) {
    impl_.setClientDisconnectedHandler([handler = std::move(handler)](auto& connection) {
        return handler(*connection);
    });
}

int QuasarServer::listenTcpLocal(int port) {
    return impl_.init(port);
}

void QuasarServer::listenService() {
    impl_.initService(serviceName_);
}

void QuasarServer::listenTcpHost(const std::string& hostname, int port) {
    impl_.initService(hostname, port);
}

int QuasarServer::port() const {
    return impl_.port();
}

int QuasarServer::getConnectedClientCount() const {
    return impl_.getConnectedClientCount();
}

void QuasarServer::waitConnectionsAtLeast(size_t count) {
    impl_.waitConnectionsAtLeast(count);
}

bool QuasarServer::waitConnectionsAtLeast(size_t count, std::chrono::milliseconds timeout) {
    return impl_.waitConnectionsAtLeast(count, timeout);
}

void QuasarServer::waitListening() const {
    impl_.waitListening();
}

void QuasarServer::sendToAll(const SharedMessage& message) {
    impl_.sendToAll(message.ref());
}

void QuasarServer::sendToAll(quasar::proto::QuasarMessage&& message) {
    impl_.sendToAll(message);
}

void QuasarServer::shutdown() {
    impl_.shutdown();
}
