#include "quasar_server_impl.h"

#include "length_value_tokenizer.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/threading/lifetime.h>

#include <yandex_io/protos/quasar_proto.pb.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("datacratic_ipc");

using namespace quasar;
using namespace quasar::ipc::detail::datacratic;

QuasarServerImpl::QuasarServerImpl(Lifetime::Tracker tracker, std::shared_ptr<YandexIO::Configuration> configuration, std::shared_ptr<ICallbackQueue> callbackQueue)
    : configuration_(std::move(configuration))
    , tracker_(std::move(tracker))
    , callbackQueue_(std::move(callbackQueue))
    , serviceName_(std::make_shared<std::string>("unnamed"))
{
    getTokenizer = [=]() {
        return std::make_shared<LengthValueTokenizer>();
    };

    setPollingMode(MIN_CPU_POLLING);

    onRequest = std::bind(&QuasarServerImpl::handleMessage, this, std::placeholders::_1, std::placeholders::_2);
}

QuasarServerImpl::QuasarServerImpl(std::shared_ptr<YandexIO::Configuration> configuration)
    : QuasarServerImpl(Lifetime::Tracker{}, std::move(configuration), nullptr)
{
}

QuasarServerImpl::QuasarServerImpl(const std::string& serviceName, std::shared_ptr<YandexIO::Configuration> configuration)
    : QuasarServerImpl(std::move(configuration))
{
    initService(serviceName);
}

QuasarServerImpl::QuasarServerImpl(const Lifetime& lifetime, std::shared_ptr<YandexIO::Configuration> configuration, std::shared_ptr<ICallbackQueue> callbackQueue)
    : QuasarServerImpl(lifetime.tracker(), std::move(configuration), std::move(callbackQueue))
{
}

QuasarServerImpl::~QuasarServerImpl()
{
    shutdown();
}

void QuasarServerImpl::setMessageHandler(MessageHandler handler)
{
    onQuasarMessage_ = std::move(handler);
}

void QuasarServerImpl::setClientConnectedHandler(ClientHandler handler) {
    if (callbackQueue_) {
        handler = makeClientHandlerForCallbackQueue(std::move(handler));
    }
    TCPEndpoint::setClientConnectedHandler(std::move(handler));
}

void QuasarServerImpl::setClientDisconnectedHandler(ClientHandler handler) {
    if (callbackQueue_) {
        handler = makeClientHandlerForCallbackQueue(std::move(handler));
    }
    TCPEndpoint::setClientDisconnectedHandler(std::move(handler));
}

void QuasarServerImpl::initService(const std::string& serviceName)
{
    try {
        auto serviceConfig = configuration_->getServiceConfig(serviceName);
        std::string bind = tryGetString(serviceConfig, "bind", "localhost");
        const int port = tryGetInt(serviceConfig, "port", -1);
        // do not allow to start server with default port
        Y_VERIFY(port != -1, "Fail to create Quasar Server server with name \"%s\"", serviceName.c_str());
        serviceName_ = std::make_shared<std::string>(serviceName);
        init(Datacratic::PortRange{port}, bind);
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("DatacraticServer.InitService.Exception", "Fail to start QuasarServerImpl \"" << serviceName << "\": " << ex.what());
        throw;
    } catch (...) {
        YIO_LOG_ERROR_EVENT("DatacraticServer.InitService.UnknownError", "Fail to start QuasarServerImpl \"" << serviceName << "\": unexpected exception");
        throw;
    }
}

void QuasarServerImpl::initService(const std::string& hostname, int port)
{
    try {
        serviceName_ = std::make_shared<std::string>(hostname + ":" + std::to_string(port));
        init(Datacratic::PortRange{port}, hostname);
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("DatacraticServer.InitService.Exception", "Fail to start QuasarServerImpl \"" << *serviceName_ << "\": " << ex.what());
        throw;
    } catch (...) {
        YIO_LOG_ERROR_EVENT("DatacraticServer.InitService.UnknownError", "Fail to start QuasarServerImpl \"" << *serviceName_ << "\": unexpected exception");
        throw;
    }
}

void QuasarServerImpl::sendToAll(const quasar::proto::QuasarMessage& message)
{
    TCPEndpoint::sendToAll(LengthValueTokenizer::getLengthValue(message.SerializeAsString()));
}

void QuasarServerImpl::send(const quasar::proto::QuasarMessage& message, TCPConnectionHandler* handler)
{
    handler->send(LengthValueTokenizer::getLengthValue(message.SerializeAsString()));
}

QuasarServerImpl::ClientHandler QuasarServerImpl::makeClientHandlerForCallbackQueue(ClientHandler handler)
{
    return
        [this, handler{std::move(handler)}](const std::shared_ptr<TCPConnectionHandler>& tcpConnectionHandler)
    {
        std::weak_ptr<TCPConnectionHandler> tcpConnectionHandlerWeak = tcpConnectionHandler;
        callbackQueue_->add(
            [handler, serviceName = serviceName_, tcpConnectionHandlerWeak{std::move(tcpConnectionHandlerWeak)}] {
                if (auto h = tcpConnectionHandlerWeak.lock()) {
                    try {
                        handler(h);
                    } catch (const std::exception& ex) {
                        YIO_LOG_ERROR_EVENT("DatacraticServer.ConnectionHandler.Exception", "Unhandled exception in quasar server \"" << *serviceName << "\": " << ex.what());
                        throw;
                    } catch (...) {
                        YIO_LOG_ERROR_EVENT("DatacraticServer.ConnectionHandler.UnknownError", "Unexpected exception in quasar server \"" << *serviceName << "\"");
                        throw;
                    }
                }
            }, tracker_);
    };
}

void QuasarServerImpl::handleMessage(const std::string& message, const std::shared_ptr<TCPConnectionHandler>& tcpConnectionHandler)
{
    quasar::proto::QuasarMessage protoMessage;
    if (!protoMessage.ParseFromString(TString(message)))
    {
        YIO_LOG_ERROR_EVENT("DatacraticServer.HandleMessage.ProtoParseError", "Cannot parse QuasarMessage.");
        tcpConnectionHandler->doError("Cannot parse QuasarMessage");
        return;
    }
    SharedMessage sharedMessage(std::move(protoMessage));

    if (onQuasarMessage_) {
        if (callbackQueue_) {
            std::weak_ptr<TCPConnectionHandler> tcpConnectionHandlerWeak = tcpConnectionHandler;
            callbackQueue_->add(
                [serviceName = serviceName_, onQuasarMessage = onQuasarMessage_, sharedMessage, tcpConnectionHandlerWeak{std::move(tcpConnectionHandlerWeak)}]() mutable {
                    if (auto h = tcpConnectionHandlerWeak.lock()) {
                        try {
                            onQuasarMessage(sharedMessage, h.get());
                        } catch (const std::exception& ex) {
                            YIO_LOG_ERROR_EVENT("DatacraticServer.OnQuasarMessage.Exception", "Unhandled exception in quasar server \"" << *serviceName << "\": " << ex.what());
                            throw;
                        } catch (...) {
                            YIO_LOG_ERROR_EVENT("DatacraticServer.OnQuasarMessage.UnknownError", "Unexpected exception in quasar server \"" << *serviceName << "\"");
                            throw;
                        }
                    }
                }, tracker_);
        } else {
            onQuasarMessage_(sharedMessage, tcpConnectionHandler.get());
        }
    }
}
