#pragma once

#include "tokenizer.h"

#include <yandex_io/libs/threading/steady_condition_variable.h>

#include <yandex_io/external_libs/datacratic/soa/service/endpoint.h>
#include <yandex_io/external_libs/datacratic/soa/service/transport.h>

#include <queue>
#include <thread>

namespace quasar::ipc::detail::datacratic {

    struct ConnectManager;
    struct TCPConnector: public Datacratic::EndpointBase {
        using OnConnectionError = std::function<void(const std::string&)>;
        using OnConnect = std::function<void()>;
        using OnDisconnect = std::function<void()>;

        TCPConnector();
        explicit TCPConnector(const std::string& name);

        void setConnectHandler(OnConnect handler);
        void setDisconnectHandler(OnDisconnect handler);
        void setConnectionErrorHandler(OnConnectionError handler);

        void init(std::string host, int port, const std::string& threadName = "");

        bool sendMessage(std::string data);

        bool isConnected() const;
        void waitUntilConnected() const;
        void waitUntilDisconnected() const;

        template <class Rep, class Period>
        bool waitUntilConnected(const std::chrono::duration<Rep, Period>& duration) const {
            std::unique_lock<std::mutex> lock(connectionLock_);

            std::chrono::time_point<std::chrono::steady_clock> endTimePoint;
            if (std::chrono::duration<Rep, Period>::max() == duration) {
                endTimePoint = std::chrono::steady_clock::now() + std::chrono::hours(10 * 24 * 365); // Due to bug in condition_variable: https://gcc-bugs.gcc.gnu.narkive.com/LBRgUQhD/bug-c-58931-new-condition-variable-wait-until-overflowed-by-large-time-point-steady-clock
            } else {
                endTimePoint = std::chrono::steady_clock::now() + duration;
            }
            while (nullptr == connection_)
            {
                auto waitResult = connectionCond_.wait_until(lock, endTimePoint);
                if (std::cv_status::timeout == waitResult) {
                    break;
                }
            }

            return connection_ != nullptr;
        }

        template <class Rep, class Period>
        bool waitUntilDisconnected(const std::chrono::duration<Rep, Period>& duration) const {
            std::unique_lock<std::mutex> lock(connectionLock_);

            auto endTimePoint = std::chrono::steady_clock::now() + duration;
            while (connection_ != nullptr)
            {
                auto waitResult = connectionCond_.wait_until(lock, endTimePoint);
                if (std::cv_status::timeout == waitResult) {
                    break;
                }
            }

            return nullptr == connection_;
        }

        std::string hostname() const override {
            return host_;
        }

        int port() const override {
            return port_;
        }

        void shutdown();

        virtual ~TCPConnector();

        TCPConnector(const TCPConnector& other) = delete;
        TCPConnector& operator=(const TCPConnector& other) = delete;

    protected:
        using TokenizerFactory = std::function<std::shared_ptr<Tokenizer>()>;

        void doError(const std::string& errorMessage);

        TokenizerFactory getTokenizer;

        struct TCPConnectionHandler: public Datacratic::PassiveConnectionHandler,
                                     public std::enable_shared_from_this<TCPConnectionHandler> {
            void handleData(const std::string& data) override;
            void handleError(const std::string& message) override;
            void handleDisconnect() override;
            void onGotTransport() override;
            void handleTimeout(Datacratic::Date time, size_t cookie) override;

        private:
            std::shared_ptr<Tokenizer> tokenizer_;
            std::string current_;
            friend struct TCPConnector;
        };

        friend struct ConnectManager;
        friend struct TCPConnectionHandler;

        bool sendMessageUnlocked(std::string data);
        std::shared_ptr<TCPConnectionHandler> getConnection() const;

        mutable std::mutex connectionLock_;
        mutable quasar::SteadyConditionVariable connectionCond_;
        std::shared_ptr<TCPConnectionHandler> connection_;

    private:
        void closePeer() override;

        void startConnecting();
        void startReconnecting();
        int createNonBlockingTCPSocket() const;

        void onConnect(Datacratic::TransportBase& transport);
        void onDisconnectThreadUnsafe();

        // Returns function to be call after disconnect.
        virtual std::function<void()> doBeforeDisconnect(std::shared_ptr<TCPConnectionHandler> connection);
        /** Tell the endpoint that a connection has been closed. */
        void
        notifyCloseTransport(const std::shared_ptr<Datacratic::TransportBase>& transport) override;

        void stopReconnectingThread(bool disableReconnect);

        virtual void handleMessageReceived(const std::string& response) = 0;

        std::string host_;
        int port_ = -1;

        std::mutex reconnectThreadLock_;
        std::atomic_bool reconnectThreadStopped_{false};
        quasar::SteadyConditionVariable reconnectThreadCond_;
        std::unique_ptr<std::thread> reconnectThread_;
        bool dontStartReconnect_ = false;

        std::string testName_;

        OnConnectionError onConnectionError_;
        OnConnect connectHandler_;
        OnDisconnect disconnectHandler_;
    };

} // namespace quasar::ipc::detail::datacratic
