#include "tcp_endpoint.h"

#include "length_value_tokenizer.h"

#include <yandex_io/libs/logging/logging.h>

#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;
using namespace quasar::ipc::detail::datacratic;
using namespace Datacratic;

TCPEndpoint::TCPEndpoint()
    : PassiveEndpointT<SocketTransport>("TCPEndpoint")
{
    getTokenizer = []() {
        return std::make_shared<LineTokenizer>();
    };

    onMakeNewHandler = [=]()
    {
        auto handler = std::make_shared<TCPConnectionHandler>();
        handler->tokenizer_ = getTokenizer();
        std::weak_ptr<TCPConnectionHandler> handlerWeakPtr = handler;
        auto onToken = [=](const std::string& response) {
            auto handler = handlerWeakPtr.lock();
            if (handler) {
                handler->handleRequest(response);
            }
        };
        handler->tokenizer_->onToken = onToken;

        return handler;
    };
}

std::shared_ptr<ipc::IServer::IClientConnection> TCPConnectionHandler::share()
{
    return shared_from_this();
}

void TCPConnectionHandler::unsafeSendBytes(std::string_view data)
{
    send(std::string(data));
}

void TCPConnectionHandler::scheduleClose()
{
    closeWhenHandlerFinished();
}

void TCPEndpoint::setClientConnectedHandler(ClientHandler handler)
{
    onClientConnected_ = std::move(handler);
}

void TCPEndpoint::setClientDisconnectedHandler(ClientHandler handler)
{
    onClientDisconnected_ = std::move(handler);
}

void TCPEndpoint::sendToAll(const std::string& message)
{
    std::lock_guard<std::mutex> lock(connectedClientsMutex_);

    for (auto& handler : connectedClients_)
    {
        handler->send(message);
    }
}

int TCPEndpoint::getConnectedClientCount() const {
    std::lock_guard<std::mutex> lock(connectedClientsMutex_);
    return connectedClients_.size();
}

void TCPEndpoint::shutdown()
{
    closePeer();
    PassiveEndpoint::shutdown();
}

TCPEndpoint::~TCPEndpoint()
{
    shutdown();
}

bool TCPEndpoint::waitConnectionsAtLeast(size_t count, std::chrono::milliseconds timeout)
{
    std::unique_lock<std::mutex> lock(connectedClientsMutex_);
    return connectionsCondVar_.wait_for(lock, timeout, [this, count]() {
        return connectedClients_.size() >= count;
    });
}

void TCPEndpoint::waitConnectionsAtLeast(size_t count)
{
    std::unique_lock<std::mutex> lock(connectedClientsMutex_);
    connectionsCondVar_.wait(lock, [this, count]() {
        return connectedClients_.size() >= count;
    });
}

void TCPEndpoint::addHandler(const std::shared_ptr<TCPConnectionHandler>& handler)
{
    std::lock_guard<std::mutex> lock(connectedClientsMutex_);
    connectedClients_.insert(handler);
    connectionsCondVar_.notify_all();
}

void TCPEndpoint::removeHandler(const std::shared_ptr<TCPConnectionHandler>& handler)
{
    std::lock_guard<std::mutex> lock(connectedClientsMutex_);
    connectedClients_.erase(handler);
    connectionsCondVar_.notify_all();
}

void TCPConnectionHandler::send(const SharedMessage& message)
{
    send(LengthValueTokenizer::getLengthValue(message->SerializeAsString()));
}

void TCPConnectionHandler::send(proto::QuasarMessage&& message)
{
    send(LengthValueTokenizer::getLengthValue(message.SerializeAsString()));
}

void TCPConnectionHandler::handleData(const std::string& data)
{
    Y_VERIFY(tokenizer_);
    tokenizer_->pushData(data);
}

void TCPConnectionHandler::handleRequest(const std::string& request)
{
    if (endpoint_->onRequest) {
        endpoint_->onRequest(request, shared_from_this());
    }
}

void TCPConnectionHandler::handleError(const std::string& errorMessage)
{
    YIO_LOG_DEBUG(errorMessage);
}

void TCPConnectionHandler::onGotTransport()
{
    Y_VERIFY(dynamic_cast<TCPEndpoint*>(transport().get_endpoint()) != nullptr);
    endpoint_ = static_cast<TCPEndpoint*>(transport().get_endpoint());
    auto sharedThis = shared_from_this();
    endpoint_->addHandler(sharedThis);

    if (endpoint_->onClientConnected_) {
        endpoint_->onClientConnected_(sharedThis);
    }

    startReading();
}

void TCPConnectionHandler::onDisassociate()
{
    auto sharedThis = shared_from_this();
    endpoint_->removeHandler(sharedThis);
    if (endpoint_->onClientDisconnected_) {
        endpoint_->onClientDisconnected_(sharedThis);
    }
}

void TCPConnectionHandler::setLocalData(std::shared_ptr<void> data)
{
    localData_ = std::move(data);
}

std::shared_ptr<void> TCPConnectionHandler::getLocalData() const {
    return localData_;
}
