#pragma once

#include "tokenizer.h"

#include <yandex_io/libs/ipc/i_server.h>

#include <yandex_io/libs/threading/steady_condition_variable.h>

#include <yandex_io/external_libs/datacratic/soa/service/passive_endpoint.h>
#include <yandex_io/protos/quasar_proto_forward.h>

#include <memory>
#include <unordered_set>

namespace quasar::ipc::detail::datacratic {

    class TCPEndpoint;
    class TCPConnectionHandler: public ipc::IServer::IClientConnection,
                                public Datacratic::PassiveConnectionHandler,
                                public std::enable_shared_from_this<TCPConnectionHandler> {
    public:
        using PassiveConnectionHandler::send;

        void send(const SharedMessage& message) override;
        void send(Message&& message) override;
        void unsafeSendBytes(std::string_view data) override;
        std::shared_ptr<IClientConnection> share() override;
        void scheduleClose() override;

        void handleData(const std::string& data) override;
        void handleError(const std::string& errorMessage) override;

        void handleRequest(const std::string& request);

        void onGotTransport() override;
        void onDisassociate() override;

        void setLocalData(std::shared_ptr<void> data); // Thread unsafe
        std::shared_ptr<void> getLocalData() const;    // Thread unsafe

    private:
        TCPEndpoint* endpoint_ = nullptr;
        std::shared_ptr<Tokenizer> tokenizer_;

        std::shared_ptr<void> localData_;

        friend class TCPEndpoint;
    };

    class TCPEndpoint: public Datacratic::PassiveEndpointT<Datacratic::SocketTransport> {
    public:
        using ClientHandler = std::function<void(const std::shared_ptr<TCPConnectionHandler>& handler)>;

        TCPEndpoint();

        void shutdown();

        ~TCPEndpoint();

        void sendToAll(const std::string& message);

        int getConnectedClientCount() const;
        /**
         * @brief Wait until there will be 'count' connections to endpoint
         */
        void waitConnectionsAtLeast(size_t count);
        bool waitConnectionsAtLeast(size_t count, std::chrono::milliseconds timeout);

        virtual void setClientConnectedHandler(ClientHandler handler);
        virtual void setClientDisconnectedHandler(ClientHandler handler);

        TCPEndpoint(const TCPEndpoint&) = delete;
        TCPEndpoint& operator=(const TCPEndpoint&) = delete;

    protected:
        void addHandler(const std::shared_ptr<TCPConnectionHandler>& handler);
        void removeHandler(const std::shared_ptr<TCPConnectionHandler>& handler);

        mutable std::mutex connectedClientsMutex_;
        std::unordered_set<std::shared_ptr<TCPConnectionHandler>> connectedClients_;

        using RequestHandler = std::function<void(const std::string&, const std::shared_ptr<TCPConnectionHandler>&)>;
        RequestHandler onRequest;

        using TokenizerFactory = std::function<std::shared_ptr<Tokenizer>()>;
        TokenizerFactory getTokenizer;

        friend class TCPConnectionHandler;

    private:
        SteadyConditionVariable connectionsCondVar_;

        ClientHandler onClientConnected_;
        ClientHandler onClientDisconnected_;
    };
} // namespace quasar::ipc::detail::datacratic
