#pragma once

#include <chrono>
#include <queue>
#include <set>
#include <unordered_map>

namespace quasar::ipc::detail::datacratic {

    /******************************************************************************/
    /* TIMEOUT MAP                                                                */
    /******************************************************************************/

    template <typename Key, typename Value>
    struct TimeoutMap {
        size_t size() const {
            return map.size();
        }

        bool count(const Key& key) const {
            return map.count(key);
        }

        Value& get(const Key& key)
        {
            auto it = map.find(key);
            if (map.end() == it) {
                throw std::runtime_error("key not present in the timeout map.");
            }
            return it->second.value;
        }

        const Value& get(const Key& key) const {
            auto it = map.find(key);
            if (map.end() == it) {
                throw std::runtime_error("key not present in the timeout map.");
            }
            return it->second.value;
        }

        bool tryGet(const Key& key, Value& value) const {
            auto it = map.find(key);
            if (map.end() == it) {
                return false;
            }
            value = it->second.value;

            return true;
        }

        bool emplace(Key key, Value value, std::chrono::milliseconds timeout)
        {
            auto deadline = std::chrono::steady_clock::now() + std::chrono::milliseconds(timeout);
            auto ret = map.insert(std::make_pair(std::move(key), Entry(std::move(value), deadline)));
            if (!ret.second) {
                return false;
            }

            queue.emplace(ret.first->first, deadline);
            return true;
        }

        Value pop(const Key& key)
        {
            auto it = map.find(key);
            if (map.end() == it) {
                throw std::runtime_error("key not present in the timeout map.");
            }

            Value value = std::move(it->second.value);
            map.erase(it);
            return value;
        }

        bool erase(const Key& key)
        {
            return map.erase(key);
        }

        template <typename Fn>
        size_t expire(Fn& fn)
        {
            auto now = std::chrono::steady_clock::now();
            std::vector<std::pair<Key, Entry>> toExpire;
            toExpire.reserve(1 << 4);

            while (!queue.empty() && queue.top().timeout <= now) {
                TimeoutEntry entry = std::move(queue.top());
                queue.pop();

                auto it = map.find(entry.key);
                if (it == map.end()) {
                    continue;
                }
                if (it->second.timeout > now) {
                    continue;
                }

                toExpire.emplace_back(std::move(*it));
                map.erase(it);
            }

            for (auto& entry : toExpire) {
                fn(std::move(entry.first), std::move(entry.second.value));
            }

            return toExpire.size();
        }

        std::unordered_map<Key, Value> move()
        {
            while (!queue.empty()) {
                queue.pop();
            }

            std::unordered_map<Key, Value> result;
            for (auto& entry : map)
            {
                result[std::move(entry.first)] = std::move(entry.second.value);
            }
            map.clear();
            return result;
        }

        void forEach(std::function<void(const Key&, const Value&)> handler) const {
            for (const auto& keyValue : map) {
                handler(keyValue.first, keyValue.second.value);
            }
        }

    private:
        using TimePoint = std::chrono::steady_clock::time_point;

        struct Entry {
            Value value;
            TimePoint timeout;

            Entry(Value value, TimePoint timeout)
                : value(std::move(value))
                , timeout(timeout)
            {
            }
        };

        struct TimeoutEntry {
            Key key;
            TimePoint timeout;

            TimeoutEntry(Key key, TimePoint timeout)
                : key(std::move(key))
                , timeout(timeout)
            {
            }

            bool operator<(const TimeoutEntry& other) const {
                return timeout > other.timeout;
            }
        };

        std::unordered_map<Key, Entry> map;
        std::priority_queue<TimeoutEntry> queue;
    };

} // namespace quasar::ipc::detail::datacratic
