#pragma once

#include "message.h"

#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/libs/threading/lifetime.h>

#include <yandex_io/protos/quasar_proto_forward.h>

#include <chrono>
#include <functional>
#include <memory>
#include <string>

namespace quasar::ipc {

    class IConnector {
    public:
        using MessageHandler = std::function<void(const SharedMessage& quasarMessage)>;
        using OnConnectionError = std::function<void(const std::string&)>;
        using OnConnect = std::function<void()>;
        using OnDisconnect = std::function<void()>;

        using OnDone = std::function<void(const SharedMessage&)>;
        using OnError = std::function<void(const std::string&)>;

        virtual ~IConnector();

        virtual const std::string& serviceName() const = 0;

        // Set options/callbacks
        virtual void setConnectHandler(OnConnect handler) = 0;
        virtual void setDisconnectHandler(OnDisconnect handler) = 0;
        virtual void setConnectionErrorHandler(OnConnectionError handler) = 0;
        virtual void setMessageHandler(MessageHandler handler) = 0;
        virtual void setSilentMode(bool silentMode) = 0;

        // EventBus mode
        virtual bool sendMessage(const SharedMessage& message) = 0;

        // Request-Reply mode
        virtual void sendRequest(UniqueMessage&& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout) = 0;
        virtual SharedMessage sendRequestSync(UniqueMessage&& message, std::chrono::milliseconds timeout) = 0;

        YIO_IPC_DEPRECATED_MESSAGE_API virtual bool sendMessage(Message&& message) = 0;
        YIO_IPC_DEPRECATED_MESSAGE_API virtual SharedMessage sendRequestSync(Message&& message, std::chrono::milliseconds timeout) = 0;
        YIO_IPC_DEPRECATED_MESSAGE_API virtual void sendRequest(Message&& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout) = 0;

        virtual bool tryConnectToService() = 0;
        virtual void connectToService() = 0;
        virtual void connectToTcpHost(const std::string& hostname, int port) = 0;
        virtual void connectToTcpLocal(int port) = 0;

        virtual bool isConnected() const = 0;
        virtual void waitUntilConnected() const = 0;
        virtual void waitUntilDisconnected() const = 0;
        virtual bool waitUntilConnected(const std::chrono::milliseconds& duration) const = 0;
        virtual bool waitUntilDisconnected(const std::chrono::milliseconds& duration) const = 0;

        virtual void shutdown() = 0;
    };

} // namespace quasar::ipc
