#pragma once

#include <yandex_io/protos/quasar_proto_forward.h>

#include <memory>

#ifdef YIO_IPC_NO_DEPRECATED
    #define YIO_IPC_DEPRECATED_MESSAGE_API [[deprecated]]
#else
    #define YIO_IPC_DEPRECATED_MESSAGE_API
#endif

namespace quasar::ipc {

    /**
     * There are 3 types that represent messages:
     *
     * `UniqueMessage`: a unique mutable by-reference wrapper
     *  - Mutable: value can be modified
     *  - Non-copyable (move only): only one mutable reference to a single must exist at one time.
     *  - Can be destructively converted into an immutable `SharedMessage`
     *
     * `SharedMessage` a shared immutable by-reference wrapper
     *  - Immutable: value cannot be modified
     *  - Copyable: all copies share the same actual value
     *
     * `Message`: the bare protobuf message.
     *  - Normal const qualifiers apply
     *  - It can be quite large, so copies (and even moves) should be avoided.
     *  - Usage in IPC API is deprecated
     *
     * The preferred way to create a message is to use a builder:
     *
     * ```
     * // SharedMessage
     * auto message = ipc::buildMessage([](auto& msg) {
     *     // here go protobuf mutators on `Message msg`
     * });
     *
     * // UniqueMessage
     * auto mutableMessage = ipc::buildUniqueMessage([](auto& msg) {
     *     // here go protobuf mutators on `Message msg`
     * });
     * ```
     *
     **/

    // Alias to protobuf message: "value" type
    using Message = proto::QuasarMessage;

    // unique_ptr-like mutable allocated value
    class UniqueMessage {
        friend class SharedMessage;

    public:
        UniqueMessage() = default;
        UniqueMessage(UniqueMessage&&) = default;
        UniqueMessage(const UniqueMessage&) = delete;

        explicit UniqueMessage(const Message& message)
            : UniqueMessage(newMessage(message))
        {
        }

        explicit UniqueMessage(Message&& message)
            : UniqueMessage(newMessage(std::move(message)))
        {
        }

        static UniqueMessage create() {
            return UniqueMessage(newMessage());
        }

        UniqueMessage& operator=(UniqueMessage&&) = default;
        UniqueMessage& operator=(const UniqueMessage&) = delete;

        UniqueMessage& operator=(std::nullptr_t) {
            ptr_.reset();
            return *this;
        }

        bool operator!() const {
            return !ptr_;
        }
        explicit operator bool() const {
            return !!ptr_;
        }

        const Message* operator->() const {
            return ptr_.get();
        }
        const Message& operator*() const {
            return *ptr_;
        }
        const Message& ref() const {
            return *ptr_;
        }

        Message* operator->() {
            return ptr_.get();
        }
        Message& operator*() {
            return *ptr_;
        }
        Message& ref() {
            return *ptr_;
        }

    private:
        // Internal ctor: it lets user violate the `unique reference` invariant
        explicit UniqueMessage(std::shared_ptr<Message> messagePtr)
            : ptr_(std::move(messagePtr))
        {
        }

    private:
        // These methods call std::make_shared<Message>() and require its definition
        // By moving them away from header, we can keep the header free from `*.pb.h` includes
        static std::shared_ptr<Message> newMessage();
        static std::shared_ptr<Message> newMessage(const Message& /*msg*/);
        static std::shared_ptr<Message> newMessage(Message&& /*msg*/);

    private:
        // NB: shared_ptr, even though UniqueMessage suggests unique ownership
        // Reason: UniqueMessage is usually destined to be "frozen" as a SharedMessage
        // Using a shared_ptr lets us allocate with an internal control block via std::make_shared()
        std::shared_ptr<Message> ptr_;
    };

    // shared_ptr-like immutable allocated value
    class SharedMessage {
    public:
        SharedMessage() = default;
        SharedMessage(const SharedMessage&) = default;
        SharedMessage(SharedMessage&&) = default;

        SharedMessage(UniqueMessage&& message)
            : ptr_(std::move(message.ptr_))
        {
        }

        explicit SharedMessage(const Message& message)
            : SharedMessage(UniqueMessage(message))
        {
        }

        explicit SharedMessage(Message&& message)
            : SharedMessage(UniqueMessage(std::move(message)))
        {
        }

        SharedMessage& operator=(const SharedMessage&) = default;
        SharedMessage& operator=(SharedMessage&&) = default;

        SharedMessage& operator=(UniqueMessage&& message) {
            ptr_ = std::move(message.ptr_);
            return *this;
        }
        SharedMessage& operator=(std::nullptr_t) {
            ptr_.reset();
            return *this;
        }

        bool operator!() const {
            return !ptr_;
        }
        explicit operator bool() const {
            return !!ptr_;
        }

        const Message* operator->() const {
            return ptr_.get();
        }
        const Message& operator*() const {
            return *ptr_;
        }
        const Message& ref() const {
            return *ptr_;
        }

        std::shared_ptr<const Message> share() const {
            return ptr_;
        }

    private:
        std::shared_ptr<const Message> ptr_;
    };

    template <typename F>
    SharedMessage buildMessage(F&& builder) {
        auto msg = UniqueMessage::create();
        builder(*msg);
        return SharedMessage(std::move(msg));
    }

    template <typename F>
    UniqueMessage buildUniqueMessage(F&& builder) {
        auto msg = UniqueMessage::create();
        builder(*msg);
        return msg;
    }

} // namespace quasar::ipc

std::ostream& operator<<(std::ostream& os, const quasar::ipc::Message& message);
std::ostream& operator<<(std::ostream& os, const quasar::ipc::UniqueMessage& message);
std::ostream& operator<<(std::ostream& os, const quasar::ipc::SharedMessage& message);
