#include "mixed_connector.h"

#include <yandex_io/libs/ipc/helpers.h>

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>

#include <yandex_io/protos/quasar_proto.pb.h>

#include <future>

YIO_DEFINE_LOG_MODULE("mixed_ipc");

using namespace quasar;
using namespace quasar::ipc;
using namespace quasar::ipc::detail::mixed;

class MixedConnector::MixedLocalConnection: public IMixedServer::LocalConnection,
                                            public std::enable_shared_from_this<MixedConnector::MixedLocalConnection> {
public:
    MixedLocalConnection(MixedConnector& owner, Lifetime::Tracker tracker)
        : owner_(owner)
        , tracker_(std::move(tracker))
    {
    }

    bool shutdown() {
        bool expected = false;
        return shutdown_.compare_exchange_strong(expected, true);
    }

public: // IMixedServer::LocalConnection
    void onConnect() override {
        if (auto lock = tracker_.lock()) {
            owner_.localConnectionOnConnect(this);
        }
    }

    void onDisconnect() override {
        bool expected = false;
        if (!shutdown_.compare_exchange_strong(expected, true)) {
            return;
        }
        if (auto lock = tracker_.lock()) {
            owner_.localConnectionOnDisconnect(this);
        }
    }

    std::shared_ptr<IClientConnection> share() override {
        return shared_from_this();
    }

    void send(const SharedMessage& message) override {
        if (shutdown_) {
            return;
        }
        if (auto lock = tracker_.lock()) {
            owner_.localConnectionReceiveFromServer(this, message);
        }
    }

    void send(Message&& message) override {
        send(SharedMessage{std::move(message)});
    }

    void scheduleClose() override {
        if (auto lock = tracker_.lock()) {
            owner_.localConnectionScheduleClose(this);
        }
    }

private:
    MixedConnector& owner_;
    Lifetime::Tracker tracker_;
    std::atomic<bool> shutdown_;
};

MixedConnector::MixedConnector(std::shared_ptr<YandexIO::Configuration> configuration, std::string serviceName, IpcConnectorFactory ipcConnectorFactory)
    : MixedConnector(std::move(configuration), std::move(serviceName), std::move(ipcConnectorFactory), nullptr)
{
}

MixedConnector::MixedConnector(std::shared_ptr<YandexIO::Configuration> configuration, std::string serviceName, IpcConnectorFactory ipcConnectorFactory, std::shared_ptr<ICallbackQueue> callbackQueue)
    : configuration_(std::move(configuration))
    , serviceName_(std::move(ipc::helpers::checkServiceName(serviceName)))
    , ipcConnectorFactory_(std::move(ipcConnectorFactory))
    , callbackQueue_(std::move(callbackQueue))
{
}

MixedConnector::~MixedConnector()
{
    shutdown();
    if (!callbackQueue_->isWorkingThread()) {
        callbackQueue_->wait(ICallbackQueue::AwatingType::EGOIST);
    }
    lifetime_.die();
}

void MixedConnector::bind(const std::shared_ptr<IMixedServer>& mixedServer)
{
    if (!mixedServer) {
        unbind();
        return;
    }

    std::shared_ptr<IMixedServer> oldMixedServer;
    std::lock_guard lock(mutex_);
    if (shutdown_) {
        return;
    }

    if (mixedServer->serviceName() != serviceName_) {
        throw std::runtime_error("The service name on the connector does not match the name of the local server");
    }

    oldMixedServer = mixedServer_.lock();
    if (oldMixedServer == mixedServer) {
        return;
    }

    if (localConnection_) {
        if (oldMixedServer) {
            oldMixedServer->removeLocalConnection(localConnection_);
        }
        localConnection_ = nullptr;
    }

    ipcConnectorFactory_ = nullptr;
    ipcConnector_ = nullptr;

    if (started_) {
        localConnection_ = std::make_shared<MixedLocalConnection>(*this, lifetime_.tracker());
        mixedServer->addLocalConnection(localConnection_);
    }
    mixedServer_ = mixedServer;
}

void MixedConnector::unbind()
{
    std::shared_ptr<IMixedServer> mixedServer;
    std::lock_guard lock(mutex_);
    mixedServer = mixedServer_.lock();
    unbindUnsafe(mixedServer);
}

bool MixedConnector::isLocal() const {
    std::lock_guard lock(mutex_);
    return !ipcConnector_;
}

void MixedConnector::logMessages(bool lm) {
    logMessages_ = lm;
}

const std::string& MixedConnector::serviceName() const {
    return serviceName_;
}

void MixedConnector::setConnectHandler(OnConnect handler) {
    std::lock_guard lock(mutex_);
    ensureNotStaredUnsafe();
    connectHandler_ = std::move(handler);
}

void MixedConnector::setDisconnectHandler(OnDisconnect handler) {
    std::lock_guard lock(mutex_);
    ensureNotStaredUnsafe();
    disconnectHandler_ = std::move(handler);
}

void MixedConnector::setConnectionErrorHandler(OnConnectionError handler)
{
    std::lock_guard lock(mutex_);
    ensureNotStaredUnsafe();
    connectionErrorHandler_ = std::move(handler);
}

void MixedConnector::setMessageHandler(MessageHandler handler)
{
    std::lock_guard lock(mutex_);
    ensureNotStaredUnsafe();
    messageHandler_ = std::move(handler);
}

void MixedConnector::setSilentMode(bool silentMode)
{
    std::lock_guard lock(mutex_);
    ensureNotStaredUnsafe();
    silentMode_ = silentMode;
}

bool MixedConnector::sendMessage(Message&& message)
{
    return sendMessage(SharedMessage(std::move(message)));
}

bool MixedConnector::sendMessage(const SharedMessage& message)
{
    std::shared_ptr<IMixedServer> mixedServer;
    std::lock_guard lock(mutex_);
    YIO_LOG_TRACE("Connector " << serviceName_ << ".[" << this << "].sendMessage: " << shortUtf8DebugString(message.ref()));
    mixedServer = mixedServer_.lock();
    return sendMessageUnsafe(mixedServer, message);
}

void MixedConnector::sendRequest(Message&& m, OnDone onDone, OnError onError, std::chrono::milliseconds timeout)
{
    sendRequest(makeUUID(), UniqueMessage(std::move(m)), std::move(onDone), std::move(onError), timeout);
}

void MixedConnector::sendRequest(UniqueMessage&& m, OnDone onDone, OnError onError, std::chrono::milliseconds timeout)
{
    sendRequest(makeUUID(), std::move(m), std::move(onDone), std::move(onError), timeout);
}

SharedMessage MixedConnector::sendRequestSync(Message&& message, std::chrono::milliseconds timeout)
{
    return sendRequestSync(UniqueMessage(std::move(message)), timeout);
}

SharedMessage MixedConnector::sendRequestSync(UniqueMessage&& message, std::chrono::milliseconds timeout)
{
    std::unique_lock lock(mutex_);
    if (!callbackQueue_) {
        throw std::runtime_error("Connector \"" + serviceName_ + "\" not configured");
    }

    if (callbackQueue_->isWorkingThread()) {
        throw std::runtime_error("Can't send sync request in \"" + serviceName_ + "\" connector working thread this will completely block it work");
    }

    std::promise<void> requestDone;
    SharedMessage result;
    auto onDone = [&](const SharedMessage& response) {
        result = response;
        requestDone.set_value();
    };

    auto onError = [&](const std::string& errorMessage) {
        requestDone.set_exception(std::make_exception_ptr(
            std::runtime_error("Cannot send request: " + errorMessage)));
    };
    lock.unlock();

    auto sharedMessage = sendRequest(makeUUID(), std::move(message), std::move(onDone), std::move(onError), timeout);
    auto future = requestDone.get_future();
    constexpr std::chrono::seconds DEADLOCK_TIMEOUT{2};
    auto ready = future.wait_for(timeout + DEADLOCK_TIMEOUT);
    if (ready == std::future_status::ready) {
        future.get();
        return result;
    }

    // DEADLOCK situation
    extractRequest(sharedMessage->request_id());
    throw std::logic_error("The sendRequestSync method failed to complete within the specified time, possibly DEADLOCK in connector \"" + serviceName_ + "\": " + convertMessageToJsonString(*sharedMessage));
}

bool MixedConnector::tryConnectToService()
{
    if (configuration_ && configuration_->getFullConfig().isMember(serviceName_)) {
        connectToService();
        return true;
    }
    YIO_LOG_INFO("There is no \"" << serviceName_ << "\" entry in config");
    return false;
}

void MixedConnector::connectToService()
{
    doConnect(ServiceAuto{});
}

void MixedConnector::connectToTcpHost(const std::string& /*hostname*/, int /*port*/)
{
    throw std::runtime_error("Unsupported feature for mixed connector");
}

void MixedConnector::connectToTcpLocal(int port)
{
    doConnect(ServiceTcpLocal{port});
}

bool MixedConnector::isConnected() const {
    std::shared_ptr<IMixedServer> mixedServer;
    std::lock_guard lock(mutex_);
    if (localConnection_) {
        mixedServer = mixedServer_.lock();
        if (mixedServer) {
            return mixedServer->isStarted();
        }
        return false;
    }

    if (ipcConnector_) {
        return ipcConnector_->isConnected();
    }
    return false;
}

void MixedConnector::waitUntilConnected() const {
    waitUntilConnected(std::chrono::seconds{0x7FFFFFFF});
}

void MixedConnector::waitUntilDisconnected() const {
    waitUntilDisconnected(std::chrono::seconds{0x7FFFFFFF});
}

bool MixedConnector::waitUntilConnected(const std::chrono::milliseconds& timeout) const {
    /*
     * I have no idea how to do this normally without reworking the ipc::IConnector interface
     */
    const auto until = std::chrono::steady_clock::now() + timeout;
    do {
        if (isConnected()) {
            return true;
        }
        std::this_thread::sleep_for(std::chrono::milliseconds{20});
    } while (!shutdown_ && until > std::chrono::steady_clock::now());
    return isConnected();
}

bool MixedConnector::waitUntilDisconnected(const std::chrono::milliseconds& timeout) const {
    /*
     * I have no idea how to do this normally without reworking the ipc::IConnector interface
     */
    const auto until = std::chrono::steady_clock::now() + timeout;
    do {
        if (!isConnected()) {
            return true;
        }
        std::this_thread::sleep_for(std::chrono::milliseconds{20});
    } while (!shutdown_ && until > std::chrono::steady_clock::now());
    return !isConnected();
}

void MixedConnector::shutdown()
{
    std::shared_ptr<IMixedServer> mixedServer;
    std::lock_guard lock(mutex_);
    shutdown_ = true;
    mixedServer = mixedServer_.lock();
    unbindUnsafe(mixedServer);
    if (ipcConnector_) {
        ipcConnector_->shutdown();
    }
}

void MixedConnector::doConnect(const Service& service)
{
    const auto* pServiceAuto = std::get_if<ServiceAuto>(&service);
    const auto* pServiceTcpLocal = std::get_if<ServiceTcpLocal>(&service);

    std::shared_ptr<IMixedServer> mixedServer;
    std::lock_guard lock(mutex_);
    ensureNotStaredUnsafe();

    mixedServer = mixedServer_.lock();

    std::shared_ptr<ICallbackQueue> callbackQueue = callbackQueue_;
    std::shared_ptr<ipc::IConnector> ipcConnector;

    try {
        if (!callbackQueue) {
            callbackQueue = std::make_shared<NamedCallbackQueue>("MixedConnector:" + serviceName_);
        }

        if (!mixedServer && ipcConnectorFactory_) {
            ipcConnector = ipcConnectorFactory_(serviceName_, lifetime_, callbackQueue);
            ipcConnector->setSilentMode(silentMode_);
            ipcConnector->setConnectHandler(
                [tracker = lifetime_.tracker(), this]() {
                    if (auto lock = tracker.lock()) {
                        connectHandler();
                    }
                });
            ipcConnector->setDisconnectHandler(
                [tracker = lifetime_.tracker(), this]() {
                    if (auto lock = tracker.lock()) {
                        disconnectHandler();
                    }
                });
            ipcConnector->setConnectionErrorHandler(
                [tracker = lifetime_.tracker(), this](const std::string& text) {
                    if (auto lock = tracker.lock()) {
                        connectionErrorHandler(text);
                    }
                });
            ipcConnector->setMessageHandler(
                [tracker = lifetime_.tracker(), this](const SharedMessage& message) {
                    if (auto lock = tracker.lock()) {
                        messageHandler(message);
                    }
                });

            if (pServiceAuto) {
                ipcConnector->connectToService();
            } else if (pServiceTcpLocal) {
                ipcConnector->connectToTcpLocal(pServiceTcpLocal->port);
            } else {
                throw std::runtime_error("Unexpected variant value in MixedConnector for " + serviceName_);
            }
        }
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("MixedIpcConnector.Connect.Exception", "Fail to initialize mixed connector \"" << serviceName_ << "\": " << ex.what());
        throw;
    } catch (...) {
        YIO_LOG_ERROR_EVENT("MixedIpcConnector.Connect.UnknownError", "Fail to initialize mixed connector \"" << serviceName_ << "\": unexpected exception");
        throw;
    }

    ipcConnectorFactory_ = nullptr;
    callbackQueue_ = callbackQueue;
    ipcConnector_ = ipcConnector;
    started_ = true;
    if (mixedServer) {
        localConnection_ = std::make_shared<MixedLocalConnection>(*this, lifetime_.tracker());
        mixedServer->addLocalConnection(localConnection_);
    }
}

SharedMessage MixedConnector::sendRequest(std::string uuid, UniqueMessage&& m, OnDone onDone, OnError onError, std::chrono::milliseconds timeout)
{
    std::shared_ptr<IMixedServer> mixedServer;
    std::lock_guard lock(mutex_);
    if (!callbackQueue_) {
        throw std::runtime_error("Connector \"" + serviceName_ + "\" not configured");
    }

    auto until = std::chrono::steady_clock::now() + timeout;
    m->set_request_id(std::move(uuid));

    YIO_LOG_TRACE("Connector " << serviceName_ << ".[" << this << "].sendRequest: " << shortUtf8DebugString(*m));
    requests_.push_back(
        Request{
            .requestId = m->request_id(),
            .onDone = std::move(onDone),
            .onError = std::move(onError),
            .until = until,
        });
    mixedServer = mixedServer_.lock();
    SharedMessage sharedMessage{std::move(m)};
    if (sendMessageUnsafe(mixedServer, sharedMessage)) {
        callbackQueue_->addDelayed([this] { checkExpiredRequests(); }, timeout, lifetime_);
    }
    return sharedMessage;
}

void MixedConnector::ensureNotStaredUnsafe()
{
    if (shutdown_) {
        YIO_LOG_ERROR_EVENT("MixedIpcConnector.StartWhileShutdown", "Service connector \"" << serviceName_ << "\""
                                                                                           << " already shutdown and can't be configured");
        throw std::runtime_error("Service connector \"" + serviceName_ + "\"" + " already shutdown and can't be configured");
    }

    if (started_) {
        YIO_LOG_ERROR_EVENT("MixedIpcConnector.StartWhileAlreadyStarted", "Service connector \"" << serviceName_ << "\""
                                                                                                 << " already started and can't be configured");
        throw std::runtime_error("Service connector \"" + serviceName_ + "\"" + " already started and can't be configured");
    }
}

bool MixedConnector::sendMessageUnsafe(const std::shared_ptr<IMixedServer>& mixedServer, const SharedMessage& message)
{
    if (logMessages_) {
        if (localConnection_) {
            YIO_LOG_DEBUG("Mixed connector " << this << " \"" << serviceName_ << "\" (srv=" << mixedServer.get() << ") SEND: " << message);
        } else {
            YIO_LOG_DEBUG("Mixed connector " << this << " \"" << serviceName_ << "\" (remote) SEND: " << message);
        }
    }

    if (localConnection_) {
        return localConnectionSendToServerUnsafe(mixedServer, message);
    }

    if (ipcConnector_ && ipcConnector_->isConnected()) {
        if (ipcConnector_->sendMessage(message)) {
            return true;
        }
        if (!message->request_id().empty()) {
            notifyRequestError(message->request_id(), "Failed to send a message to the \"" + serviceName_ + "\" service because the connection was lost");
        }
    } else {
        if (!message->request_id().empty()) {
            notifyRequestError(message->request_id(), "Unable to send a message to the \"" + serviceName_ + "\" service because there is no connection");
        }
    }

    YIO_LOG_DEBUG("Send message failed to service " << serviceName_);
    return false;
}

void MixedConnector::checkExpiredRequests()
{
    Y_ENSURE_THREAD(callbackQueue_);
    std::list<Request> expired;
    {
        std::lock_guard lock(mutex_);
        auto now = std::chrono::steady_clock::now();
        for (auto it = requests_.begin(); it != requests_.end();) {
            if (it->until < now) {
                expired.emplace_back(std::move(*it));
                it = requests_.erase(it);
            } else {
                ++it;
            }
        }
    }

    for (auto& request : expired) {
        try {
            request.onDone = nullptr;
            if (request.onError) {
                request.onError("The waiting time for a response to a request to the \"" + serviceName_ + "\" service has expired");
            }
        } catch (const std::exception& ex) {
            YIO_LOG_ERROR_EVENT("MixedIpcConnector.RequestExpiredOnError.Exception", "Unexpected exception in expired request: " << ex.what());
        } catch (...) {
            YIO_LOG_ERROR_EVENT("MixedIpcConnector.RequestExpiredOnError.UnknownError", "Unexpected and undefined exception in expired request");
        }
    }
}

void MixedConnector::connectHandler()
{
    if (!connectHandler_) {
        return;
    }

    if (!callbackQueue_->isWorkingThread()) {
        callbackQueue_->add([this] { connectHandler(); }, lifetime_);
        return;
    }

    Y_ENSURE_THREAD(callbackQueue_);
    connectHandler_();
}

void MixedConnector::disconnectHandler() {
    if (!callbackQueue_->isWorkingThread()) {
        callbackQueue_->add([this] { disconnectHandler(); }, lifetime_);
        return;
    }

    Y_ENSURE_THREAD(callbackQueue_);

    std::list<Request> expired;
    {
        std::lock_guard lock(mutex_);
        expired = std::move(requests_);
    }

    for (auto& request : expired) {
        try {
            request.onDone = nullptr;
            if (request.onError) {
                request.onError("Connection to server " + serviceName_ + " lost");
            }
        } catch (const std::exception& ex) {
            YIO_LOG_ERROR_EVENT("MixedIpcConnector.RequestExpiredOnError.Exception", "Unexpected exception in expired request: " << ex.what());
        } catch (...) {
            YIO_LOG_ERROR_EVENT("MixedIpcConnector.RequestExpiredOnError.UnknownError", "Unexpected and undefined exception in expired request");
        }
    }

    if (disconnectHandler_) {
        disconnectHandler_();
    }
}

void MixedConnector::connectionErrorHandler(const std::string& text)
{
    if (!connectionErrorHandler_) {
        return;
    }

    if (!callbackQueue_->isWorkingThread()) {
        callbackQueue_->add([this, text] { connectionErrorHandler(text); }, lifetime_);
        return;
    }

    Y_ENSURE_THREAD(callbackQueue_);
    connectionErrorHandler_(text);
}

void MixedConnector::messageHandler(const SharedMessage& message)
{
    if (!messageHandler_ && !message->has_request_id()) {
        return;
    }

    if (!callbackQueue_->isWorkingThread()) {
        callbackQueue_->add(
            [this, message]() {
                messageHandler(message);
            }, lifetime_);
        return;
    }

    Y_ENSURE_THREAD(callbackQueue_);

    if (message->has_request_id()) {
        auto request = extractRequest(message->request_id());
        if (request) {
            request->onError = nullptr;
            if (request->onDone) {
                request->onDone(message);
            }
            return;
        } else {
            YIO_LOG_WARN("Cannot find handler for response with id: '" << message->request_id() << "'");
        }
    }

    if (logMessages_) {
        if (localConnection_) {
            std::lock_guard lock(mutex_);
            YIO_LOG_DEBUG("Mixed connector " << this << " \"" << serviceName_ << "\" (srv=" << mixedServer_.lock().get() << ") RECV: " << message);
        } else {
            YIO_LOG_DEBUG("Mixed connector " << this << " \"" << serviceName_ << "\" (remote) RECV: " << message);
        }
    }

    if (messageHandler_) {
        messageHandler_(message);
    }
}

std::optional<MixedConnector::Request> MixedConnector::extractRequest(const std::string& requestId)
{
    std::optional<Request> result;

    std::unique_lock lock(mutex_);
    auto it = std::find_if(requests_.begin(), requests_.end(),
                           [&](const auto& request) {
                               return requestId == request.requestId;
                           });
    if (it != requests_.end()) {
        result.emplace(std::move(*it));
        requests_.erase(it);
    }
    return result;
}

void MixedConnector::notifyRequestError(std::string requestId, std::string text)
{
    /*
     * Error reporting should always occur asynchronously and on the connector thread
     */
    callbackQueue_->add(
        [this, requestId{std::move(requestId)}, text{std::move(text)}] {
            Y_ENSURE_THREAD(callbackQueue_);
            auto request = extractRequest(requestId);
            if (request) {
                request->onDone = nullptr;
                if (request->onError) {
                    request->onError(text);
                }
            }
        }, lifetime_);
}

void MixedConnector::unbindUnsafe(const std::shared_ptr<IMixedServer>& mixedServer)
{
    if (mixedServer) {
        if (localConnection_) {
            auto localConnection = localConnection_;
            bool manualShutdown = localConnection_->shutdown();
            mixedServer->removeLocalConnection(localConnection_);
            localConnection_ = nullptr;
            if (manualShutdown) {
                localConnectionOnDisconnect(localConnection.get());
            }
        }
    }
    mixedServer_.reset();
}

void MixedConnector::localConnectionOnConnect(MixedLocalConnection* /*unused*/) {
    callbackQueue_->add(
        [this] {
            connectHandler();
        }, lifetime_);
}

void MixedConnector::localConnectionOnDisconnect(MixedLocalConnection* /*unused*/) {
    callbackQueue_->add(
        [this] {
            disconnectHandler();
        }, lifetime_);
}

void MixedConnector::localConnectionReceiveFromServer(MixedLocalConnection* /*unused*/, const SharedMessage& message)
{
    callbackQueue_->add(
        [this, message] {
            messageHandler(message);
        }, lifetime_);
}

void MixedConnector::localConnectionScheduleClose(MixedLocalConnection* connection) {
    Y_ENSURE_NOT_THREAD(callbackQueue_);
    callbackQueue_->add(
        [this, connection] {
            std::shared_ptr<IMixedServer> mixedServer;
            std::unique_lock lock(mutex_);
            if (connection == localConnection_.get()) {
                mixedServer = mixedServer_.lock();
                unbindUnsafe(mixedServer);
            }
        }, lifetime_);
}

bool MixedConnector::localConnectionSendToServerUnsafe(const std::shared_ptr<IMixedServer>& mixedServer, const SharedMessage& message)
{
    if (!mixedServer) {
        if (message->has_request_id()) {
            notifyRequestError(message->request_id(), "The \"" + serviceName_ + "\" service expired");
        }
        return false;
    } else if (!mixedServer->isStarted()) {
        if (message->has_request_id()) {
            notifyRequestError(message->request_id(), "The \"" + serviceName_ + "\" service disconnected");
        }
        return false;
    }

    try {
        mixedServer->messageFromLocalConnection(message, *localConnection_);
    } catch (const std::exception& ex) {
        if (message->has_request_id()) {
            notifyRequestError(message->request_id(), "Caught an exception when trying to send a message to the \"" + serviceName_ + "\" service: " + ex.what());
        }
        return false;
    } catch (...) {
        if (message->has_request_id()) {
            notifyRequestError(message->request_id(), "Caught an unexpected exception when trying to send a message to the \"" + serviceName_ + "\" service");
        }
        return false;
    }
    return true;
}
