#pragma once

#include "i_mixed_server.h"

#include <yandex_io/libs/ipc/i_connector.h>

#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/device/i_device.h>

#include <list>
#include <optional>
#include <variant>

namespace quasar::ipc::detail::mixed {

    class MixedConnector final: public ipc::IConnector {
    public:
        using IpcConnectorFactory = std::function<std::shared_ptr<ipc::IConnector>(
            std::string serviceName,
            const quasar::Lifetime& lifetime,
            const std::shared_ptr<ICallbackQueue>& callbackQueue)>;

        MixedConnector(std::shared_ptr<YandexIO::Configuration> configuration, std::string serviceName, IpcConnectorFactory ipcConnectorFactory);
        MixedConnector(std::shared_ptr<YandexIO::Configuration> configuration, std::string serviceName, IpcConnectorFactory ipcConnectorFactory, std::shared_ptr<ICallbackQueue> callbackQueue);
        ~MixedConnector();

        void bind(const std::shared_ptr<IMixedServer>& mixedServer);
        void unbind();

        bool isLocal() const;
        void logMessages(bool /*lm*/);

    public: // ipc::IConnector
        const std::string& serviceName() const override;
        void setConnectHandler(OnConnect handler) override;
        void setDisconnectHandler(OnDisconnect handler) override;
        void setConnectionErrorHandler(OnConnectionError handler) override;
        void setMessageHandler(MessageHandler handler) override;
        void setSilentMode(bool silentMode) override;
        bool sendMessage(const SharedMessage& message) override;
        bool sendMessage(Message&& message) override;
        void sendRequest(UniqueMessage&& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout) override;
        void sendRequest(Message&& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout) override;
        SharedMessage sendRequestSync(UniqueMessage&& message, std::chrono::milliseconds timeout) override;
        SharedMessage sendRequestSync(Message&& message, std::chrono::milliseconds timeout) override;
        bool tryConnectToService() override;
        void connectToService() override;
        void connectToTcpHost(const std::string& hostname, int port) override;
        void connectToTcpLocal(int port) override;
        bool isConnected() const override;
        void waitUntilConnected() const override;
        void waitUntilDisconnected() const override;
        bool waitUntilConnected(const std::chrono::milliseconds& timeout) const override;
        bool waitUntilDisconnected(const std::chrono::milliseconds& timeout) const override;
        void shutdown() override;

    private:
        class MixedLocalConnection;
        struct ServiceAuto {};
        struct ServiceTcpLocal {
            int port;
        };
        using Service = std::variant<ServiceAuto, ServiceTcpLocal>;
        void doConnect(const Service& service);

        struct Request {
            std::string requestId;
            OnDone onDone;
            OnError onError;
            std::chrono::steady_clock::time_point until;
        };

        SharedMessage sendRequest(std::string uuid, UniqueMessage&& m, OnDone onDone, OnError onError, std::chrono::milliseconds timeout);
        void ensureNotStaredUnsafe();
        bool sendMessageUnsafe(const std::shared_ptr<IMixedServer>& mixedServer, const SharedMessage& message);
        void checkExpiredRequests();
        void connectHandler();
        void disconnectHandler();
        void connectionErrorHandler(const std::string& text);
        void messageHandler(const SharedMessage& message);
        std::optional<Request> extractRequest(const std::string& requestId);
        void notifyRequestError(std::string requestId, std::string text);
        void unbindUnsafe(const std::shared_ptr<IMixedServer>& mixedServer);

        void localConnectionOnConnect(MixedLocalConnection* connection);
        void localConnectionOnDisconnect(MixedLocalConnection* connection);
        void localConnectionReceiveFromServer(MixedLocalConnection* connection, const SharedMessage& message);
        void localConnectionScheduleClose(MixedLocalConnection* connection);
        bool localConnectionSendToServerUnsafe(const std::shared_ptr<IMixedServer>& mixedServer, const SharedMessage& message);

    private:
        Lifetime lifetime_;
        const std::shared_ptr<YandexIO::Configuration> configuration_;
        const std::string serviceName_;
        IpcConnectorFactory ipcConnectorFactory_;

        std::atomic<bool> logMessages_{false};

        mutable std::mutex mutex_;
        std::shared_ptr<ICallbackQueue> callbackQueue_;
        bool started_{false};
        bool shutdown_{false};

        std::weak_ptr<IMixedServer> mixedServer_;
        std::shared_ptr<MixedLocalConnection> localConnection_;
        std::shared_ptr<ipc::IConnector> ipcConnector_;

        OnConnect connectHandler_;
        OnDisconnect disconnectHandler_;
        OnConnectionError connectionErrorHandler_;
        MessageHandler messageHandler_;
        bool silentMode_{false};
        std::list<Request> requests_;
    };

} // namespace quasar::ipc::detail::mixed
