#include "mixed_ipc_factory.h"

#include "mixed_connector.h"
#include "mixed_server.h"

#include <yandex_io/libs/ipc/asio/asio_ipc_factory.h>
#include <yandex_io/libs/ipc/datacratic/public.h>

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <iomanip>
#include <optional>

using namespace quasar;
using namespace quasar::ipc;
using namespace quasar::ipc::detail::mixed;

namespace {
    std::optional<MixedIpcFactory::Transport> stringToTransport(const std::string& text) {
        if (text == "net" || text == "NET") {
            return MixedIpcFactory::Transport::NET;
        } else if (text == "local" || text == "LOCAL") {
            return MixedIpcFactory::Transport::LOCAL;
        } else if (text == "mixed" || text == "MIXED") {
            return MixedIpcFactory::Transport::MIXED;
        } else {
            return std::nullopt;
        }
    }

    std::string_view transportToString(MixedIpcFactory::Transport transport) {
        switch (transport) {
            case MixedIpcFactory::Transport::DEFAULT:
                return "DEFAULT";
            case MixedIpcFactory::Transport::NET:
                return "NET";
            case MixedIpcFactory::Transport::LOCAL:
                return "LOCAL";
            case MixedIpcFactory::Transport::MIXED:
                return "MIXED";
        }
        return "UNKNOWN";
    }

    class DefaultCallbackQueueCreator {
    public:
        DefaultCallbackQueueCreator() = default;
        std::shared_ptr<ICallbackQueue> operator()(const std::string& serviceName, MixedIpcFactory::Purpose purpose, bool /*localOnly*/) {
            switch (purpose) {
                case MixedIpcFactory::Purpose::FACTORY:
                    return std::make_shared<NamedCallbackQueue>(serviceName);
                case MixedIpcFactory::Purpose::MIXED_SERVER:
                    return std::make_shared<NamedCallbackQueue>("MixedServer:" + serviceName);
                case MixedIpcFactory::Purpose::MIXED_CONNECTOR:
                    return std::make_shared<NamedCallbackQueue>("MixedConnector:" + serviceName);
                case MixedIpcFactory::Purpose::NET_SERVER:
                    return std::make_shared<NamedCallbackQueue>("NetServer:" + serviceName);
                case MixedIpcFactory::Purpose::NET_CONNECTOR:
                    return std::make_shared<NamedCallbackQueue>("NetConnector:" + serviceName);
            }
            throw std::runtime_error("Fail to create callback queue: Undefined MixedIpcFactory::Purpose");
        }
    };
} // namespace

std::shared_ptr<MixedIpcFactory::Context> MixedIpcFactory::createDefaultContext(std::shared_ptr<YandexIO::Configuration> configuration)
{
    auto context = std::make_shared<MixedIpcFactory::Context>();
    context->configuration = configuration;
    context->callbackQueueCreator = DefaultCallbackQueueCreator{};
    context->factoryCallbackQueue = std::make_shared<NamedCallbackQueue>("MixedIpcFactory");

    const auto& common = configuration->getCommonConfig();
    if (common.isMember("ipc")) {
        const auto& ipc = common["ipc"];
        if (ipc.isMember("netImpl")) {
            if (ipc["netImpl"] == "asio") {
                context->netImplementation = NetImplementation::ASIO;
            }
        }
    }

    return context;
}

MixedIpcFactory::MixedIpcFactory(std::shared_ptr<Context> context)
    : MixedIpcFactory(std::move(context), "")
{
}

MixedIpcFactory::MixedIpcFactory(std::shared_ptr<Context> context, std::string processName)
    : context_(std::move(context))
    , processName_(std::move(processName))
{
    Y_VERIFY(context_);
    Y_VERIFY(context_->configuration);

    if (!context_->callbackQueueCreator) {
        context_->callbackQueueCreator = DefaultCallbackQueueCreator{};
    }

    // NET or MIXED (LOCAL + NET)
    if (context_->netImplementation == NetImplementation::ASIO) {
        auto factory = std::make_shared<ipc::AsioIpcFactory>(context_->configuration);
        context_->customNetServerCreator =
            [factory](std::string serviceName, const Lifetime::Tracker& /*tracker*/, const std::shared_ptr<ICallbackQueue>& /*callbackQueue*/) {
                return factory->createIpcServer(serviceName);
            };
        context_->customNetConnectorCreator =
            [factory](std::string serviceName, const Lifetime::Tracker& /*tracker*/, const std::shared_ptr<ICallbackQueue>& /*callbackQueue*/) {
                return factory->createIpcConnector(serviceName);
            };
    } else if (context_->netImplementation == NetImplementation::LEGACY) {
        context_->customNetServerCreator =
            [configuration{context_->configuration}](std::string serviceName, const Lifetime::Tracker& /*tracker*/, const std::shared_ptr<ICallbackQueue>& /*cbQueue*/) {
                return ipc::datacratic::createIpcServer(std::move(serviceName), configuration);
            };
        context_->customNetConnectorCreator =
            [configuration{context_->configuration}](std::string serviceName, const Lifetime::Tracker& /*tracker*/, const std::shared_ptr<ICallbackQueue>& /*cbQueue*/) {
                return ipc::datacratic::createIpcConnector(std::move(serviceName), configuration);
            };
    } else {
        // CUSTOM
        if (!context_->customNetServerCreator || !context_->customNetConnectorCreator) {
            throw std::runtime_error("NET implementation is required and set to CUSTOM, but some custom net callbacks are empty");
        }
    }

    if (context_->factoryCallbackQueue) {
        context_->factoryCallbackQueue->addDelayed([this] { dumpStatistic(); }, std::chrono::seconds{20}, lifetime_);
    }
}

MixedIpcFactory::~MixedIpcFactory() {
    lifetime_.die();
}

std::shared_ptr<MixedIpcFactory::Context> MixedIpcFactory::context() const {
    return context_;
}

std::string MixedIpcFactory::processName() const {
    std::lock_guard lock(mutex_);
    return processName_;
}

void MixedIpcFactory::setProcessName(std::string processName)
{
    std::lock_guard lock(mutex_);
    if (!processName_.empty()) {
        auto text = "Duplicate setting of the process name, possibly a design error";
        YIO_LOG_WARN(text);
        throw std::runtime_error(text);
    }
    processName_ = std::move(processName);
}

std::shared_ptr<IServer> MixedIpcFactory::createIpcServer(IMixedServer::ListenerWeak listener, const std::string& serviceName, Transport transport, Purpose purpose, std::shared_ptr<ICallbackQueue> callbackQueue) {
    std::unique_lock lock(mutex_);
    if (transport == Transport::DEFAULT) {
        transport = serviceTransport(serviceName);
    }

    if (transport == Transport::NET) {
        auto server = cretePureNetServerUnsafe(serviceName, transport);
        return server;
    }

    auto localOnly = (transport == Transport::LOCAL);
    if (purpose == Purpose::MIXED_SERVER && !localOnly) {
        int servicePort = context_->configuration->getServicePort(serviceName);
        localOnly = (servicePort <= 0);
    }

    auto it = servers_.find(serviceName);
    if (it != servers_.end() && !it->second.expired()) {
        throw std::runtime_error("Duplicate server name \"" + serviceName + "\"");
    }

    if (!callbackQueue) {
        callbackQueue = context_->callbackQueueCreator(serviceName, purpose, localOnly);
    }
    auto server = std::shared_ptr<MixedServer>(
        new MixedServer(
            std::move(listener),
            serviceName,
            (localOnly ? nullptr : context_->customNetServerCreator),
            callbackQueue),
        makeMixedServerDeleter());

    servers_[serviceName] = server;
    auto jt = connectors_.find(serviceName);
    if (jt != connectors_.end()) {
        for (auto& connector : jt->second) {
            if (auto c = connector.lock()) {
                c->bind(server);
                YIO_LOG_DEBUG("Mixed connector (this=" << c.get() << ") for service \"" << processNameUnsafe() << "." << serviceName << "\" bound to server (this=" << server.get() << ")");
            }
        }
    }
    YIO_LOG_DEBUG("Create mixed server (this=" << server.get() << ") for service \"" << processNameUnsafe() << "." << serviceName << "\" "
                                               << " in thread " << callbackQueue->name() << " (" << callbackQueue.get() << ") "
                                               << "via " << transportToString(transport)
                                               << (localOnly && transport != Transport::LOCAL ? " (local)" : ""));
    return server;
}

std::shared_ptr<IConnector> MixedIpcFactory::createIpcConnector(const std::string& serviceName, Transport transport, Purpose purpose, std::shared_ptr<ICallbackQueue> callbackQueue) {
    std::unique_lock lock(mutex_);
    if (transport == Transport::DEFAULT) {
        transport = serviceTransport(serviceName);
    }

    if (transport == Transport::NET) {
        auto connector = cretePureNetConnectorUnsafe(serviceName, transport);
        return connector;
    }

    auto localOnly = (transport == Transport::LOCAL);
    if (purpose == Purpose::MIXED_CONNECTOR && !localOnly) {
        int servicePort = context_->configuration->getServicePort(serviceName);
        localOnly = (servicePort <= 0);
    }

    if (!callbackQueue) {
        callbackQueue = context_->callbackQueueCreator(serviceName, purpose, localOnly);
    }
    auto connector = std::shared_ptr<MixedConnector>(
        new MixedConnector(
            context_->configuration,
            serviceName,
            (localOnly ? nullptr : context_->customNetConnectorCreator),
            callbackQueue),
        makeMixedConnectorDeleter());
    auto sniffConnector = sniff(serviceName);
    if (sniffConnector) {
        connector->logMessages(true);
    }
    connectors_[serviceName].push_back(connector);
    YIO_LOG_DEBUG("Create mixed connector (this=" << connector.get() << ") for service \"" << processNameUnsafe() << "." << serviceName << "\""
                                                  << (sniffConnector ? " [SNIFFER]" : "")
                                                  << " in thread " << callbackQueue->name() << " (" << callbackQueue.get() << ") "
                                                  << " via " << transportToString(transport)
                                                  << (localOnly && transport != Transport::LOCAL ? " (local)" : ""));
    auto serverIt = servers_.find(serviceName);
    if (serverIt != servers_.end()) {
        if (auto server = serverIt->second.lock()) {
            connector->bind(server);
            YIO_LOG_DEBUG("Mixed connector (this=" << connector.get() << ") for service \"" << processNameUnsafe() << "." << serviceName << "\" bound to server (this=" << server.get() << ")");
        }
    }
    return connector;
}

std::shared_ptr<IServer> MixedIpcFactory::findIpcServer(const std::string& serviceName) const {
    std::lock_guard lock(mutex_);
    auto it = servers_.find(serviceName);
    return it != servers_.end() ? it->second.lock() : nullptr;
}

std::shared_ptr<IServer> MixedIpcFactory::createIpcServer(const std::string& serviceName) {
    return createIpcServer(IMixedServer::ListenerWeak{}, serviceName, Transport::DEFAULT, Purpose::MIXED_SERVER, nullptr);
}

std::shared_ptr<IConnector> MixedIpcFactory::createIpcConnector(const std::string& serviceName) {
    return createIpcConnector(serviceName, Transport::DEFAULT, Purpose::MIXED_CONNECTOR, nullptr);
}

MixedIpcFactory::Transport MixedIpcFactory::serviceTransport(const std::string& serviceName) const {
    if (!context_->configuration) {
        return context_->transport;
    }

    const auto& common = context_->configuration->getCommonConfig();
    if (common.isMember("ipc")) {
        const auto& ipc = common["ipc"];
        auto readTransport =
            [&](const char* name, Transport transport) -> std::optional<Transport> {
            if (ipc.isMember(name)) {
                const auto& node = ipc[name];
                if (node.isArray()) {
                    for (Json::ArrayIndex index = 0; index < node.size(); ++index) {
                        if (node[index] == serviceName) {
                            return transport;
                        }
                    }
                }
            }
            return std::nullopt;
        };
        if (auto t = readTransport("mixed", MixedIpcFactory::Transport::MIXED)) {
            return *t;
        }
        if (auto t = readTransport("net", MixedIpcFactory::Transport::NET)) {
            return *t;
        }
        if (auto t = readTransport("local", MixedIpcFactory::Transport::LOCAL)) {
            return *t;
        }
        if (auto optTransport = stringToTransport(tryGetString(ipc, "transport"))) {
            return *optTransport;
        }
    }

    return context_->transport;
}

bool MixedIpcFactory::sniff(const std::string& serviceName) const {
    if (context_->configuration) {
        const auto& common = context_->configuration->getCommonConfig();
        if (common.isObject() && common.isMember("ipc")) {
            const auto& ipc = common["ipc"];
            if (ipc.isObject() && ipc.isMember("sniff")) {
                const auto& sniff = ipc["sniff"];
                if (sniff.isArray()) {
                    for (Json::ArrayIndex index = 0; index < sniff.size(); ++index) {
                        if (sniff[index] == serviceName) {
                            return true;
                        }
                    }
                }
            }
        }
    }
    return false;
}

void MixedIpcFactory::cleanupExpired(const std::string& serviceName) noexcept {
    std::lock_guard lock(mutex_);
    auto serverIt = servers_.find(serviceName);
    if (serverIt != servers_.end()) {
        if (serverIt->second.expired()) {
            servers_.erase(serverIt);
        }
    }

    auto connectorsIt = connectors_.find(serviceName);
    if (connectorsIt != connectors_.end()) {
        auto& connectors = connectorsIt->second;
        for (auto jt = connectors.begin(); jt != connectors.end();) {
            if (jt->expired()) {
                jt = connectors.erase(jt);
            } else {
                ++jt;
            }
        }
        if (connectors.empty()) {
            connectors_.erase(connectorsIt);
        }
    }
}

std::shared_ptr<IServer> MixedIpcFactory::cretePureNetServerUnsafe(const std::string& serviceName, Transport transport) {
    Y_VERIFY(transport == Transport::NET);

    if (!context_->customNetServerCreator) {
        YIO_LOG_WARN("Fail to create net ipc server. Creator is not initialized");
        return nullptr;
    }
    auto p = processNameUnsafe();
    auto serverObj = context_->customNetServerCreator(serviceName, lifetime_, nullptr);
    auto serverPtr = serverObj.get();
    auto server = std::shared_ptr<IServer>(
        serverPtr,
        [serverObj{std::move(serverObj)}, p](IServer* /*unused*/) mutable {
            YIO_LOG_DEBUG("Destroy net server (this=" << serverObj.get() << ") for service \"" << p << "." << serverObj->serviceName() << "\"");
            serverObj.reset();
        });

    YIO_LOG_DEBUG("Create net only server (this=" << serverPtr << ") for service \"" << p << "." << serviceName << "\"");
    return server;
}

std::shared_ptr<IConnector> MixedIpcFactory::cretePureNetConnectorUnsafe(const std::string& serviceName, Transport transport) {
    Y_VERIFY(transport == Transport::NET);

    if (!context_->customNetConnectorCreator) {
        YIO_LOG_WARN("Fail to create net ipc connector. Creator is not initialized");
        return nullptr;
    }
    auto p = processNameUnsafe();
    auto connectorObj = context_->customNetConnectorCreator(serviceName, lifetime_, nullptr);
    auto connectorPtr = connectorObj.get();
    auto connector = std::shared_ptr<IConnector>(
        connectorPtr,
        [connectorObj{std::move(connectorObj)}, p, serviceName](IConnector* /*unused*/) mutable {
            YIO_LOG_DEBUG("Destroy net connector (this=" << connectorObj.get() << ") for service \"" << p << "." << connectorObj->serviceName() << "\"");
            connectorObj.reset();
        });

    YIO_LOG_DEBUG("Create net only connector (this=" << connectorPtr << ") for service \"" << p << "." << serviceName << "\"");
    return connector;
}

std::function<void(MixedServer*)> MixedIpcFactory::makeMixedServerDeleter()
{
    return
        [this, tracker = lifetime_.tracker()](MixedServer* mixedServer) {
            try {
                mixedServer->shutdown();
            } catch (const std::exception& ex) {
                YIO_LOG_WARN("Unexpected exception while shutdown server " << mixedServer->serviceName() << ": " << ex.what());
            } catch (...) {
                YIO_LOG_WARN("Unexpected exception while shutdown server " << mixedServer->serviceName() << ": undefined exception");
            }
            if (auto lock = tracker.lock()) {
                auto processName = (std::lock_guard(mutex_), processNameUnsafe());
                YIO_LOG_DEBUG("Destroy mixed server (this=" << mixedServer << ") for service \"" << processName << "." << mixedServer->serviceName() << "\"");
                cleanupExpired(mixedServer->serviceName());
            }
            delete mixedServer;
        };
}

std::function<void(MixedConnector*)> MixedIpcFactory::makeMixedConnectorDeleter()
{
    return
        [this, tracker = lifetime_.tracker()](MixedConnector* connector) {
            try {
                connector->shutdown();
            } catch (const std::exception& ex) {
                YIO_LOG_WARN("Unexpected exception while shutdown connector " << connector->serviceName() << ": " << ex.what());
            } catch (...) {
                YIO_LOG_WARN("Unexpected exception while shutdown connector " << connector->serviceName() << ": undefined exception");
            }
            if (auto lock = tracker.lock()) {
                auto processName = (std::lock_guard(mutex_), processNameUnsafe());
                YIO_LOG_DEBUG("Destroy mixed connector (this=" << connector << ") for service \"" << processName << "." << connector->serviceName() << "\"");
                cleanupExpired(connector->serviceName());
            }
            delete connector;
        };
}

void MixedIpcFactory::dumpStatistic() const {
    std::lock_guard lock(mutex_);
    size_t serverCount = 0;
    size_t serverNetCount = 0;
    size_t serverLocalCount = 0;
    std::this_thread::sleep_for(std::chrono::milliseconds{1});
    for (const auto& ws : servers_) {
        const auto& n = ws.first;
        if (auto s = ws.second.lock()) {
            auto stat = s->stat();
            serverCount += 1;
            serverNetCount += stat.netClientCount;
            serverLocalCount += stat.localClientCount;
            YIO_LOG_INFO("#MXDMP " << std::setw(32) << std::left << (processNameUnsafe() + ".server." + n) << ": net=" << std::setfill(' ') << std::setw(3) << stat.netClientCount << ", local=" << std::setfill(' ') << std::setw(3) << stat.localClientCount);
            std::this_thread::sleep_for(std::chrono::milliseconds{1});
        }
    }
    YIO_LOG_INFO("@MXDMP " << std::setw(32) << std::left << (processNameUnsafe() + ".server.TOTAL") << ": net=" << std::setfill(' ') << std::setw(3) << serverNetCount << ", local=" << std::setfill(' ') << std::setw(3) << serverLocalCount << ", instances=" << std::setfill(' ') << std::setw(3) << serverCount);
    std::this_thread::sleep_for(std::chrono::milliseconds{1});

    size_t clientCount = 0;
    size_t clientNetCount = 0;
    size_t clientLocalCount = 0;
    for (const auto& wc : connectors_) {
        size_t n = 0;
        size_t l = 0;
        for (const auto& c : wc.second) {
            if (auto connector = c.lock()) {
                if (connector->isLocal()) {
                    ++l;
                    ++clientLocalCount;
                } else {
                    ++n;
                    ++clientNetCount;
                }
            }
        }
        if (l + n) {
            ++clientCount;
            YIO_LOG_INFO("#MXDMP " << std::setw(32) << std::left << (processNameUnsafe() + ".client." + wc.first) << ": net=" << std::setfill(' ') << std::setw(3) << n << ", local=" << std::setfill(' ') << std::setw(3) << l);
            std::this_thread::sleep_for(std::chrono::milliseconds{1});
        }
    }
    YIO_LOG_INFO("@MXDMP " << std::setw(32) << std::left << (processNameUnsafe() + ".client.TOTAL") << ": net=" << std::setfill(' ') << std::setw(3) << clientCount << ", local=" << std::setfill(' ') << std::setw(3) << clientNetCount << ", instances=" << std::setfill(' ') << std::setw(3) << clientLocalCount);
    std::this_thread::sleep_for(std::chrono::milliseconds{1});
}

std::string MixedIpcFactory::processNameUnsafe() const {
    return processName_.empty() ? "<unknown>" : processName_;
}
