#include "mixed_server.h"

#include <yandex_io/libs/ipc/helpers.h>

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>

#include <yandex_io/protos/quasar_proto.pb.h>

YIO_DEFINE_LOG_MODULE("mixed_ipc");

using namespace quasar;
using namespace quasar::ipc;
using namespace quasar::ipc::detail::mixed;

MixedServer::Listener::~Listener() = default;

void MixedServer::LocalConnection::unsafeSendBytes(std::string_view /*data*/)
{
    throw std::runtime_error("Unsupported feature for local connection");
}

MixedServer::MixedServer(std::string serviceName, IpcServerFactory ipcServerFactory)
    : MixedServer(ListenerWeak{}, std::move(serviceName), std::move(ipcServerFactory), nullptr)
{
}

MixedServer::MixedServer(std::string serviceName, IpcServerFactory ipcServerFactory, std::shared_ptr<ICallbackQueue> callbackQueue)
    : MixedServer(ListenerWeak{}, std::move(serviceName), std::move(ipcServerFactory), std::move(callbackQueue))
{
}

MixedServer::MixedServer(ListenerWeak listener, std::string serviceName, IpcServerFactory ipcServerFactory, std::shared_ptr<ICallbackQueue> callbackQueue)
    : listener_(std::move(listener))
    , serviceName_(std::move(ipc::helpers::checkServiceName(serviceName)))
    , ipcServerFactory_(std::move(ipcServerFactory))
    , callbackQueue_(std::move(callbackQueue))
{
}

MixedServer::~MixedServer()
{
    shutdown();
    if (!callbackQueue_->isWorkingThread()) {
        callbackQueue_->wait(ICallbackQueue::AwatingType::EGOIST);
    }
    lifetime_.die();
}

MixedServer::Stat MixedServer::stat() const {
    std::lock_guard lock(mutex_);
    Stat s;
    if (ipcServer_) {
        s.netClientCount = ipcServer_->getConnectedClientCount();
    }
    s.localClientCount = int(localConnections_ ? localConnections_->size() : 0);
    return s;
}

bool MixedServer::isStarted() const {
    std::lock_guard lock(mutex_);
    return started_ && !shutdown_;
}

void MixedServer::addLocalConnection(const std::shared_ptr<LocalConnection>& localConnection)
{
    if (!localConnection) {
        return;
    }

    std::unique_lock lock(mutex_);
    std::vector<std::weak_ptr<LocalConnection>> newLocalConnections;
    if (localConnections_) {
        newLocalConnections.reserve(localConnections_->size() + 1);
        for (const auto& weakConnection : *localConnections_) {
            if (auto strongConnection = weakConnection.lock()) {
                if (strongConnection != localConnection) {
                    newLocalConnections.push_back(weakConnection);
                }
            }
        }
    }
    newLocalConnections.emplace_back(localConnection);
    localConnections_ = std::make_shared<decltype(newLocalConnections)>(std::move(newLocalConnections));
    lock.unlock();

    if (started_ && !shutdown_) {
        clientConnectedHandler(*localConnection, true);
    }
}

void MixedServer::removeLocalConnection(const std::shared_ptr<LocalConnection>& localConnection)
{
    if (!localConnection) {
        return;
    }

    std::unique_lock lock(mutex_);
    if (!localConnections_) {
        return;
    }
    bool fExist = false;
    std::vector<std::weak_ptr<LocalConnection>> newLocalConnections;
    newLocalConnections.reserve(localConnections_->size() + 1);
    for (const auto& weakConnection : *localConnections_) {
        if (auto strongConnection = weakConnection.lock()) {
            if (strongConnection != localConnection) {
                newLocalConnections.push_back(weakConnection);
            } else {
                fExist = true;
            }
        }
    }
    localConnections_ = std::make_shared<decltype(newLocalConnections)>(std::move(newLocalConnections));
    lock.unlock();

    if (started_ && fExist) {
        clientDisconnectedHandler(*localConnection, true);
    }
}

void MixedServer::messageFromLocalConnection(const SharedMessage& message, LocalConnection& localConnection)
{
    messageHandler(message, localConnection);
}

const std::string& MixedServer::serviceName() const {
    return serviceName_;
}

void MixedServer::setMessageHandler(MessageHandler handler)
{
    std::lock_guard lock(mutex_);
    ensureNotStaredUnsafe();
    messageHandler_ = std::move(handler);
}

void MixedServer::setClientConnectedHandler(ClientHandler handler)
{
    std::lock_guard lock(mutex_);
    ensureNotStaredUnsafe();
    clientConnectedHandler_ = std::move(handler);
}

void MixedServer::setClientDisconnectedHandler(ClientHandler handler)
{
    std::lock_guard lock(mutex_);
    ensureNotStaredUnsafe();
    clientDisconnectedHandler_ = std::move(handler);
}

void MixedServer::listenService()
{
    doInitService(ServiceAuto{});
}

int MixedServer::listenTcpLocal(int port)
{
    return doInitService(ServiceTcpLocal{port});
}

void MixedServer::listenTcpHost(const std::string& hostname, int port)
{
    doInitService(ServiceTcpHost{hostname, port});
}

int MixedServer::port() const {
    std::lock_guard lock(mutex_);
    if (started_) {
        if (ipcServer_) {
            return ipcServer_->port();
        }
    }
    return 0;
}

int MixedServer::getConnectedClientCount() const {
    std::lock_guard lock(mutex_);
    int count = 0;
    if (localConnections_) {
        for (const auto& lc : *localConnections_) {
            count += (lc.expired() ? 0 : 1);
        }
    }
    if (ipcServer_) {
        count += ipcServer_->getConnectedClientCount();
    }
    return count;
}

void MixedServer::waitConnectionsAtLeast(size_t count)
{
    waitConnectionsAtLeast(count, std::chrono::seconds{0x7FFFFFFF});
}

bool MixedServer::waitConnectionsAtLeast(size_t count, std::chrono::milliseconds timeout) {
    /*
     * I have no idea how to do this normally without reworking the ipc::IServer interface
     */
    const auto until = std::chrono::steady_clock::now() + timeout;
    while (!shutdown_ && until > std::chrono::steady_clock::now()) {
        if (getConnectedClientCount() >= static_cast<int>(count)) {
            return true;
        }
        std::this_thread::sleep_for(std::chrono::milliseconds(20));
    }
    return getConnectedClientCount() >= static_cast<int>(count);
}

void MixedServer::waitListening() const {
    if (auto ipcServer = getIpcServer()) {
        ipcServer->waitListening();
    }
}

void MixedServer::sendToAll(const SharedMessage& message) {
    std::lock_guard lock(mutex_);
    if (shutdown_ || !ipcServer_ && (!localConnections_ || localConnections_->empty())) {
        return;
    }

    YIO_LOG_TRACE("Service " << serviceName_ << ".[" << this << "].sendToAll: " << shortUtf8DebugString(message.ref()));
    if (localConnections_ && !localConnections_->empty()) {
        callbackQueue_->add(
            [message, localConnections{localConnections_}] {
                for (const auto& weakLocalConnection : *localConnections) {
                    if (auto strongLocalConnection = weakLocalConnection.lock()) {
                        strongLocalConnection->send(message);
                    }
                }
            }, lifetime_);
    }
    if (ipcServer_) {
        std::weak_ptr<IServer> wIpcServer = ipcServer_;
        callbackQueue_->add(
            [message, wIpcServer] {
                if (auto sIpcServer = wIpcServer.lock()) {
                    sIpcServer->sendToAll(message);
                }
            }, lifetime_);
    }
}

void MixedServer::sendToAll(Message&& message) {
    sendToAll(SharedMessage{std::move(message)});
}

void MixedServer::shutdown()
{
    std::shared_ptr<IServer> ipcServer;
    decltype(localConnections_) localConnections;
    {
        std::lock_guard lock(mutex_);
        if (shutdown_) {
            return;
        }
        shutdown_ = true;
        ipcServer = ipcServer_;
        localConnections = std::move(localConnections_);
        localConnections_ = std::make_shared<decltype(localConnections_)::element_type>();
    }
    if (ipcServer) {
        ipcServer->shutdown();
    }

    if (started_) {
        if (auto l = listener_.lock()) {
            YIO_LOG_DEBUG("MixedServer::shutdown call onMixedServerShutdown: " << serviceName_);
            try {
                l->onMixedServerShutdown(serviceName_);
            } catch (...) {
            }
        }
        if (localConnections) {
            for (const auto& weakConnection : *localConnections) {
                if (auto strongConnection = weakConnection.lock()) {
                    clientDisconnectedHandler(*strongConnection, true);
                }
            }
        }
    }
}

int MixedServer::doInitService(const Service& service)
{
    const auto* pServiceAuto = std::get_if<ServiceAuto>(&service);
    const auto* pServiceTcpLocal = std::get_if<ServiceTcpLocal>(&service);
    const auto* pServiceTcpHost = std::get_if<ServiceTcpHost>(&service);

    int result = 0;

    std::unique_lock lock(mutex_);
    ensureNotStaredUnsafe();

    std::shared_ptr<ICallbackQueue> callbackQueue = callbackQueue_;
    std::shared_ptr<IServer> ipcServer;

    try {
        if (!callbackQueue) {
            callbackQueue = std::make_shared<NamedCallbackQueue>("MixedServer:" + serviceName_);
        }
        if (ipcServerFactory_) {
            ipcServer = ipcServerFactory_(serviceName_, lifetime_, callbackQueue);
            if (ipcServer) {
                ipcServer->setMessageHandler(
                    [tracker = lifetime_.tracker(), this](const SharedMessage& message, IClientConnection& connection) {
                        if (auto lock = tracker.lock()) {
                            messageHandler(message, connection);
                        }
                    });
                ipcServer->setClientConnectedHandler(
                    [tracker = lifetime_.tracker(), this](IClientConnection& connection) {
                        if (auto lock = tracker.lock()) {
                            clientConnectedHandler(connection, false);
                        }
                    });
                ipcServer->setClientDisconnectedHandler(
                    [tracker = lifetime_.tracker(), this](IClientConnection& connection) {
                        if (auto lock = tracker.lock()) {
                            clientDisconnectedHandler(connection, false);
                        }
                    });
                if (pServiceAuto) {
                    ipcServer->listenService();
                } else if (pServiceTcpLocal) {
                    result = ipcServer->listenTcpLocal(pServiceTcpLocal->port);
                } else if (pServiceTcpHost) {
                    ipcServer->listenTcpHost(pServiceTcpHost->hostname, pServiceTcpHost->port);
                } else {
                    throw std::runtime_error("Unexpected situation");
                }
            }
        }
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("MixedIpcServer.InitService.Exception", "Fail to initialize mixed server: " << ex.what());
        throw;
    } catch (...) {
        YIO_LOG_ERROR_EVENT("MixedIpcServer.InitService.UnknownError", "Fail to initialize mixed server: unexpected exception");
        throw;
    }

    ipcServerFactory_ = nullptr;
    callbackQueue_ = callbackQueue;
    ipcServer_ = ipcServer;
    started_ = true;

    if (localConnections_) {
        for (const auto& wlc : *localConnections_) {
            callbackQueue_->add([this, wlc] {
                if (auto lc = wlc.lock()) {
                    clientConnectedHandler(*lc, true);
                }
            }, lifetime_);
        }
    }
    lock.unlock();

    if (auto l = listener_.lock()) {
        YIO_LOG_DEBUG("MixedServer::doInitService call onMixedServerListen: " << serviceName_);
        try {
            l->onMixedServerListen(serviceName_);
        } catch (...) {
        }
    }
    return result;
}

std::shared_ptr<IServer> MixedServer::getIpcServer() const {
    std::lock_guard lock(mutex_);
    return ipcServer_;
}

void MixedServer::ensureNotStaredUnsafe()
{
    if (shutdown_) {
        YIO_LOG_ERROR_EVENT("MixedIpcServer.ConfigureAfterShutdown", "Service \"" << serviceName_ << "\""
                                                                                  << " already shutdown and can't be configured");
        throw std::runtime_error("Service \"" + serviceName_ + "\"" + " already shutdown and can't be configured");
    }

    if (started_) {
        YIO_LOG_ERROR_EVENT("MixedIpcServer.ConfigureAfterStart", "Service \"" << serviceName_ << "\""
                                                                               << " already started and can't be configured");
        throw std::runtime_error("Service \"" + serviceName_ + "\"" + " already started and can't be configured");
    }
}

void MixedServer::messageHandler(const SharedMessage& message, IClientConnection& connection)
{
    if (!messageHandler_) {
        return;
    }

    if (!isStarted()) {
        return;
    }

    if (!callbackQueue_->isWorkingThread()) {
        std::weak_ptr<IClientConnection> conWRef = connection.share();
        callbackQueue_->add(
            [this, message, conWRef{std::move(conWRef)}]() {
                if (auto conSRef = conWRef.lock()) {
                    messageHandler(message, *conSRef);
                }
            }, lifetime_);
        return;
    }

    Y_ENSURE_THREAD(callbackQueue_);
    messageHandler_(message, connection);
}

void MixedServer::clientConnectedHandler(IClientConnection& connection, bool notifyLocal) {
    if (!clientConnectedHandler_ && !notifyLocal) {
        return;
    }

    if (!callbackQueue_->isWorkingThread()) {
        std::weak_ptr<IClientConnection> conWRef = connection.share();
        callbackQueue_->add(
            [this, conWRef{std::move(conWRef)}, notifyLocal] {
                if (auto conSRef = conWRef.lock()) {
                    clientConnectedHandler(*conSRef, notifyLocal);
                }
            }, lifetime_);
        return;
    }

    Y_ENSURE_THREAD(callbackQueue_);
    try {
        if (clientConnectedHandler_) {
            clientConnectedHandler_(connection);
        }
    } catch (const std::exception& ex) {
        YIO_LOG_WARN("Unexpected exception in \"server." << serviceName_ << ".clientConnected(...)\": " << ex.what());
    }
    if (notifyLocal) {
        try {
            static_cast<LocalConnection&>(connection).onConnect();
        } catch (const std::exception& ex) {
            YIO_LOG_WARN("Unexpected exception in \"connector." << serviceName_ << ".connectHandler(...)\": " << ex.what());
        }
    }
}

void MixedServer::clientDisconnectedHandler(IClientConnection& connection, bool notifyLocal)
{
    if (!clientDisconnectedHandler_ && !notifyLocal) {
        return;
    }

    if (!callbackQueue_->isWorkingThread()) {
        callbackQueue_->add(
            [this, conSRef = connection.share(), notifyLocal]() mutable {
                clientDisconnectedHandler(*conSRef, notifyLocal);
            }, lifetime_);
        return;
    }

    Y_ENSURE_THREAD(callbackQueue_);
    try {
        if (clientDisconnectedHandler_) {
            clientDisconnectedHandler_(connection);
        }
    } catch (const std::exception& ex) {
        YIO_LOG_WARN("Unexpected exception in \"server." << serviceName_ << ".clientDisconnected(...)\": " << ex.what());
    }
    if (notifyLocal) {
        try {
            static_cast<LocalConnection&>(connection).onDisconnect();
        } catch (const std::exception& ex) {
            YIO_LOG_WARN("Unexpected exception in \"connector." << serviceName_ << ".disconnectHandler(...)\": " << ex.what());
        }
    }
}
