#pragma once

#include "i_mixed_server.h"

#include <variant>
#include <vector>

namespace quasar::ipc::detail::mixed {

    class MixedServer final: public IMixedServer {
    public:
        using IpcServerFactory = std::function<std::shared_ptr<IServer>(
            std::string serviceName,
            const quasar::Lifetime& lifetime,
            const std::shared_ptr<ICallbackQueue>& callbackQueue)>;

        MixedServer(std::string serviceName, IpcServerFactory ipcServerFactory);
        MixedServer(std::string serviceName, IpcServerFactory ipcServerFactory, std::shared_ptr<ICallbackQueue> callbackQueue);
        MixedServer(ListenerWeak listener, std::string serviceName, IpcServerFactory ipcServerFactory, std::shared_ptr<ICallbackQueue> callbackQueue);
        ~MixedServer();

        struct Stat {
            int netClientCount{0};
            int localClientCount{0};
        };
        Stat stat() const;

    public: // IMixedServer
        bool isStarted() const override;
        void addLocalConnection(const std::shared_ptr<LocalConnection>& localConnection) override;
        void removeLocalConnection(const std::shared_ptr<LocalConnection>& localConnection) override;
        void messageFromLocalConnection(const SharedMessage& message, LocalConnection& localConnection) override;

    public: // ipc::IServer
        const std::string& serviceName() const override;
        void setMessageHandler(MessageHandler handler) override;
        void setClientConnectedHandler(ClientHandler handler) override;
        void setClientDisconnectedHandler(ClientHandler handler) override;
        void listenService() override;
        int listenTcpLocal(int port) override;
        void listenTcpHost(const std::string& hostname, int port) override;
        int port() const override;
        int getConnectedClientCount() const override;
        void waitConnectionsAtLeast(size_t count) override;
        bool waitConnectionsAtLeast(size_t count, std::chrono::milliseconds timeout) override;
        void waitListening() const override;
        void sendToAll(const SharedMessage& message) override;
        void sendToAll(Message&& message) override;
        void shutdown() override;

    private:
        struct ServiceAuto {};
        struct ServiceTcpLocal {
            int port;
        };
        struct ServiceTcpHost {
            std::string hostname;
            int port;
        };
        using Service = std::variant<ServiceAuto, ServiceTcpLocal, ServiceTcpHost>;
        int doInitService(const Service& service);

        std::shared_ptr<IServer> getIpcServer() const;
        void ensureNotStaredUnsafe();
        void messageHandler(const SharedMessage& message, IClientConnection& connection);
        void clientConnectedHandler(IClientConnection& connection, bool notifyLocal);
        void clientDisconnectedHandler(IClientConnection& connection, bool notifyLocal);

    private:
        Lifetime lifetime_;
        const ListenerWeak listener_;
        const std::string serviceName_;
        IpcServerFactory ipcServerFactory_;

        mutable std::mutex mutex_;
        std::shared_ptr<ICallbackQueue> callbackQueue_;
        std::shared_ptr<IServer> ipcServer_;
        bool started_{false};
        bool shutdown_{false};

        MessageHandler messageHandler_;
        ClientHandler clientConnectedHandler_;
        ClientHandler clientDisconnectedHandler_;

        std::shared_ptr<const std::vector<std::weak_ptr<LocalConnection>>> localConnections_;
    };

} // namespace quasar::ipc::detail::mixed
