#pragma once
#include <yandex_io/libs/ipc/i_connector.h>

#include <yandex_io/protos/quasar_proto.pb.h>

namespace quasar::ipc::mock {

    class SimpleConnector: public quasar::ipc::IConnector {
    public:
        SimpleConnector(std::function<void(SimpleConnector*)> afterConnectToService)
            : afterConnectToService_(std::move(afterConnectToService))
        {
        }

        void pushMessage(const Message& message)
        {
            if (handler_) {
                handler_(SharedMessage{message});
            }
        }

        void pushMessage(const SharedMessage& message)
        {
            if (handler_) {
                handler_(message);
            }
        }

        void setSendMessageMethod(std::function<bool(const SharedMessage&)> m)
        {
            sendMessageMethod_ = std::move(m);
        }

        void setSendRequestSyncMethod(std::function<SharedMessage(Message&&, std::chrono::milliseconds)> m)
        {
            sendRequestSyncMethod_ = std::move(m);
        }

    public: // quasar::ipc::IConnector
        const std::string& serviceName() const override {
            throw std::runtime_error("Unexpected call of ipc::IConnector::serviceName");
        }

        void setConnectHandler(OnConnect handler) override {
            connectHandler_ = std::move(handler);
        }

        void setDisconnectHandler(OnConnect /*handler*/) override {
            throw std::runtime_error("Not implemented: " + std::string(__func__));
        }

        void setConnectionErrorHandler(OnConnectionError /*handler*/) override {
            throw std::runtime_error("Not implemented: " + std::string(__func__));
        }

        void setMessageHandler(MessageHandler handler) override {
            handler_ = std::move(handler);
        }

        void setSilentMode(bool /*silentMode*/) override {
            throw std::runtime_error("Not implemented: " + std::string(__func__));
        }

        bool sendMessage(const SharedMessage& m) override {
            if (sendMessageMethod_) {
                return sendMessageMethod_(m);
            } else {
                throw std::runtime_error("Unexpected call of ipc::IConnector::sendMessage");
            }
        }

        bool sendMessage(Message&& m) override {
            return sendMessage(SharedMessage{std::move(m)});
        }

        void sendRequest(Message&& /*message*/, OnDone /*onDone*/, OnError /*onError*/, std::chrono::milliseconds /*timeout*/) override {
            throw std::runtime_error("Not implemented: " + std::string(__func__));
        }

        void sendRequest(UniqueMessage&& /*message*/, OnDone /*onDone*/, OnError /*onError*/, std::chrono::milliseconds /*timeout*/) override {
            throw std::runtime_error("Not implemented: " + std::string(__func__));
        }

        SharedMessage sendRequestSync(Message&& m, std::chrono::milliseconds t) override {
            return sendRequestSync(UniqueMessage(std::move(m)), t);
        }

        SharedMessage sendRequestSync(UniqueMessage&& m, std::chrono::milliseconds t) override {
            if (sendRequestSyncMethod_) {
                return sendRequestSyncMethod_(std::move(*m), t);
            } else {
                throw std::runtime_error("Unexpected call of ipc::IConnector::sendRequestSync");
            }
        }

        bool tryConnectToService() override {
            connected_ = true;
            if (afterConnectToService_) {
                afterConnectToService_(this);
            }
            return connected_;
        }

        void connectToService() override {
            connected_ = true;
            if (afterConnectToService_) {
                afterConnectToService_(this);
            }
        }

        void connectToTcpHost(const std::string& /*hostname*/, int /*port*/) override {
            connected_ = true;
            if (afterConnectToService_) {
                afterConnectToService_(this);
            }
        }

        void connectToTcpLocal(int /*port*/) override {
            connected_ = true;
            if (afterConnectToService_) {
                afterConnectToService_(this);
            }
        }

        bool isConnected() const override {
            return connected_;
        }

        void waitUntilConnected() const override {
            if (connected_) {
                throw std::runtime_error("not connected");
            }
        }

        void waitUntilDisconnected() const override {
            if (connected_) {
                throw std::runtime_error("not connected");
            }
        }

        bool waitUntilConnected(const std::chrono::milliseconds& /*duration*/) const override {
            return connected_;
        }

        bool waitUntilDisconnected(const std::chrono::milliseconds& /*duration*/) const override {
            return connected_;
        }

        void shutdown() override {
            connected_ = false;
        }

    private:
        std::function<void(SimpleConnector*)> afterConnectToService_;
        std::function<bool(const SharedMessage&)> sendMessageMethod_;
        std::function<SharedMessage(Message&&, std::chrono::milliseconds)> sendRequestSyncMethod_;
        OnConnect connectHandler_;
        MessageHandler handler_;
        bool connected_{false};
    };

} // namespace quasar::ipc::mock
