#include "json_utils.h"

#include <json/json.h>

#include <exception>
#include <fstream>
#include <set>

namespace quasar {
    std::optional<Json::Value> tryParseJson(std::string_view jsonStr) {
        Json::Reader reader;
        Json::Value result;

        if (reader.parse(jsonStr.begin(), jsonStr.end(), result))
        {
            return result;
        } else {
            return std::nullopt;
        }
    }

    Json::Value tryParseJsonOrEmpty(std::string_view jsonStr)
    {
        if (auto json = tryParseJson(jsonStr)) {
            return std::move(*json);
        } else {
            return Json::Value{};
        }
    }

    Json::Value parseJson(std::string_view jsonStr)
    {
        Json::Reader reader;
        Json::Value result;
        if (reader.parse(jsonStr.begin(), jsonStr.end(), result))
        {
            return result;
        } else {
            std::string msg = "JSON Parsing error: [" +
                              reader.getFormattedErrorMessages() + "]";
            throw Json::Exception(msg);
        }
    }

    Json::Value parseJson(std::istream& ifs)
    {
        Json::Reader reader;
        Json::Value result;
        if (reader.parse(ifs, result))
        {
            return result;
        }
        std::string msg = "JSON Parsing error: [" +
                          reader.getFormattedErrorMessages() + "]";
        throw Json::Exception(msg);
    }

    Json::Value transform(const Json::Value& value, std::function<std::string(const std::string&)> f)
    {
        if (value.isString()) {
            return Json::Value(f(value.asString()));
        } else if (value.isObject()) {
            Json::Value ret;

            for (const auto& key : value.getMemberNames()) {
                ret[key] = transform(value[key], f);
            }

            return ret;
        } else if (value.isArray()) {
            Json::Value ret;

            for (const auto& item : value) {
                ret.append(transform(item, f));
            }
            return ret;
        } else {
            return value;
        }
    }

    Json::Value readJsonFromFile(const std::string& filename) {
        std::ifstream input(filename);
        if (!input.good()) {
            throw std::runtime_error(
                "Cannot open file '" + filename + "'");
        }

        return parseJson(input);
    }

    std::optional<Json::Value> tryReadJsonFromFile(const std::string& filename) {
        try {
            return readJsonFromFile(filename);
        } catch (...) {
            return std::nullopt;
        }
    }

    Json::Value getConfigFromFile(const std::string& configFile) {
        return readJsonFromFile(configFile);
    }

    Json::Value tryGetConfigFromFile(const std::string& configFile, const Json::Value& defaultConfig)
    {
        try {
            return getConfigFromFile(configFile);
        } catch (...) {
            return defaultConfig;
        }
    }

    Json::Value getJson(const Json::Value& json, const std::string& field) {
        auto value = json[field];
        if (value.isNull()) {
            throw std::runtime_error("Cannot get json field '" + field + "' from json. Field is not present");
        }
        if (!value.isObject() && !value.isArray()) {
            throw std::runtime_error("Cannot get json field '" + field + "' from json. Field is not json object");
        }

        return value;
    }

    Json::Value tryGetJson(const Json::Value& json, const std::string& field, const Json::Value& defaultValue) {
        auto value = json[field];
        if (value.isObject() || value.isArray()) {
            return value;
        }
        return defaultValue;
    }

    Json::Value getArray(const Json::Value& json, const std::string& field) {
        auto value = json[field];
        if (value.isArray()) {
            return value;
        }

        throw std::runtime_error("Cannot get json field '" + field + "' from json. Field is not array");
    }

    Json::Value tryGetArray(const Json::Value& json, const std::string& field, const Json::Value& defaultValue) {
        auto value = json[field];
        if (value.isArray()) {
            return value;
        }
        return defaultValue;
    }

    const Json::Value* findXPath(const Json::Value& json, const std::string& xpath)
    {
        const Json::Value* node = &json;
        if (xpath.empty()) {
            return node;
        }

        std::string key;
        key.reserve(128);

        size_t pos = 0;
        size_t lastPos = 0;
        while ((pos = xpath.find('/', pos)) != std::string::npos)
        {
            key.assign(xpath, lastPos, pos - lastPos);
            if (!key.empty()) {
                if (!node->isMember(key)) {
                    return nullptr;
                }
                node = &(*node)[key];
            }
            pos += 1;
            lastPos = pos;
        }
        if (lastPos != xpath.size()) {
            key.assign(xpath, lastPos);
            if (!node->isMember(key)) {
                return nullptr;
            }
            node = &(*node)[key];
        }
        return node;
    }

    Json::Value* findXPath(Json::Value& json, const std::string& xpath)
    {
        return const_cast<Json::Value*>(findXPath(const_cast<const Json::Value&>(json), xpath));
    }

    std::string getString(const Json::Value& json, const std::string& field) {
        auto value = json[field];
        if (value.isNull()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not present");
        }
        if (!value.isString()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not string");
        }

        return value.asString();
    }

    std::string tryGetString(const Json::Value& json, const std::string& field, std::string defaultValue) {
        auto value = json[field];
        if (value.isNull() || !value.isString()) {
            return defaultValue;
        }

        return value.asString();
    }

    Json::Value getArrayElement(const Json::Value& array, int index) {
        if (!array.isArray()) {
            throw std::runtime_error("Value is not an array");
        }
        if (index < 0 && (size_t)index >= array.size()) {
            throw std::out_of_range("Index out of bound");
        }
        return array[index];
    }

    int getInt(const Json::Value& json, const std::string& field) {
        auto value = json[field];
        if (value.isNull()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not present");
        }
        if (!value.isNumeric()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not numeric");
        }

        return value.asInt();
    }

    int64_t getInt64(const Json::Value& json, const std::string& field) {
        auto value = json[field];
        if (value.isNull()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not present");
        }
        if (!value.isNumeric()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not numeric");
        }

        return value.asInt64();
    }

    int64_t tryGetInt64(const Json::Value& json, const std::string& field, int64_t defaultValue) {
        auto value = json[field];
        if (value.isNull()) {
            return defaultValue;
        }
        if (!value.isNumeric()) {
            return defaultValue;
        }

        return value.asInt64();
    }

    uint64_t getUInt64(const Json::Value& json, const std::string& field) {
        auto value = json[field];
        if (value.isNull()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not present");
        }
        if (!value.isNumeric()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not numeric");
        }

        return value.asUInt64();
    }

    uint64_t tryGetUInt64(const Json::Value& json, const std::string& field, uint64_t defaultValue) {
        auto value = json[field];
        if (value.isNull()) {
            return defaultValue;
        }
        if (!value.isNumeric()) {
            return defaultValue;
        }

        return value.asUInt64();
    }

    uint32_t getUInt32(const Json::Value& json, const std::string& field) {
        auto value = json[field];
        if (value.isNull()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not present");
        }
        if (!value.isNumeric()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not numeric");
        }

        return value.asUInt();
    }

    uint32_t tryGetUInt32(const Json::Value& json, const std::string& field, uint32_t defaultValue) {
        auto value = json[field];
        if (value.isNull()) {
            return defaultValue;
        }
        if (!value.isNumeric()) {
            return defaultValue;
        }

        return value.asUInt();
    }

    uint8_t tryGetUInt8(const Json::Value& json, const std::string& field, uint8_t defaultValue) {
        auto value = json[field];
        if (value.isNull()) {
            return defaultValue;
        }
        if (!value.isNumeric()) {
            return defaultValue;
        }

        return value.asUInt();
    }

    int tryGetInt(const Json::Value& json, const std::string& field, int defaultValue) {
        auto value = json[field];
        if (value.isNull()) {
            return defaultValue;
        }
        if (!value.isNumeric()) {
            return defaultValue;
        }
        return value.asInt();
    }

    double getDouble(const Json::Value& json, const std::string& field) {
        auto value = json[field];
        if (value.isNull()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not present");
        }
        if (!value.isNumeric()) {
            throw std::runtime_error("Cannot get string field '" + field + "' from json. Field is not numeric");
        }

        return value.asDouble();
    }

    double tryGetDouble(const Json::Value& json, const std::string& field, double defaultValue) {
        auto value = json[field];
        if (value.isNull()) {
            return defaultValue;
        }
        if (!value.isNumeric()) {
            return defaultValue;
        }

        return value.asDouble();
    }

    float tryGetFloat(const Json::Value& json, const std::string& field, float defaultValue) {
        auto value = json[field];
        if (value.isNull()) {
            return defaultValue;
        }
        if (!value.isNumeric()) {
            return defaultValue;
        }

        return value.asFloat();
    }

    bool checkHasDoubleField(const Json::Value& json, const std::string& field) {
        return json.isMember(field) && json[field].isDouble();
    }

    bool getBool(const Json::Value& json, const std::string& field) {
        auto value = json[field];
        if (value.isNull()) {
            throw std::runtime_error("Cannot get bool field '" + field + "' from json. Field is not present");
        }
        if (!value.isBool()) {
            throw std::runtime_error("Cannot get bool field '" + field + "' from json. Field is not boolean");
        }

        return value.asBool();
    }

    bool tryGetBool(const Json::Value& json, const std::string& field, bool defaultValue) {
        auto value = json[field];
        if (value.isNull()) {
            return defaultValue;
        }
        if (!value.isBool()) {
            return defaultValue;
        }

        return value.asBool();
    }

    std::optional<bool> getOptionalBool(const Json::Value& json, const std::string& field) {
        auto value = json[field];
        if (value.isNull()) {
            return std::nullopt;
        }
        if (!value.isBool()) {
            return std::nullopt;
        }
        return value.asBool();
    }

    Json::Value recursiveExpandJson(Json::Value src) {
        if (src.isString()) {
            auto str = src.asString();
            if (str.length() && str[0] == '{') {
                if (auto subjson1 = quasar::tryParseJson(str)) {
                    auto subjson2 = recursiveExpandJson(std::move(*subjson1));
                    src = Json::objectValue;
                    src["_"] = std::move(subjson2);
                }
            }
        } else if (src.isArray()) {
            for (Json::ArrayIndex index = 0; index < src.size(); ++index) {
                src[index] = recursiveExpandJson(std::move(src[index]));
            }
        } else if (src.isObject()) {
            for (const auto& key : src.getMemberNames()) {
                auto& value = src[key];
                value = recursiveExpandJson(std::move(value));
            }
        }

        return src;
    }

    std::string jsonToString(const Json::Value& json, bool omitEndingLineFeed) {
        Json::FastWriter writer;
        if (omitEndingLineFeed) {
            writer.omitEndingLineFeed();
        }
        return writer.write(json);
    }

    void jsonMerge(const Json::Value& src, Json::Value& dst) {
        if (dst.isNull()) {
            dst = src;
            return;
        }

        if (!src.isObject() || !dst.isObject()) {
            return;
        }

        for (const auto& key : src.getMemberNames()) {
            // will clean destination object if source is null
            if (src[key].isNull() && dst.isMember(key)) {
                dst.removeMember(key);
                continue;
            }

            // only two objects have to be merged recursively
            if (src[key].isObject() && dst.isMember(key) && dst[key].isObject()) {
                jsonMerge(src[key], dst[key]);
            } else {
                dst[key] = src[key];
            }
        }
    }

    void jsonMergeArrays(const Json::Value& src, Json::Value& dst) {
        if (dst.isNull()) {
            dst = src;
            return;
        }

        if (src.isArray() && dst.isArray()) {
            for (const auto& item : src) {
                dst.append(item);
            }
            return;
        }

        if (!src.isObject() || !dst.isObject()) {
            return;
        }

        for (const auto& key : src.getMemberNames()) {
            // will clean destination object if source is null
            if (src[key].isNull() && dst.isMember(key)) {
                dst.removeMember(key);
                continue;
            }

            // merge arrays only on single level
            if (src[key].isArray() && dst.isMember(key) && dst[key].isArray()) {
                for (const auto& item : src[key]) {
                    dst[key].append(item);
                }
                continue;
            }

            // only two objects have to be merged recursively
            if (src[key].isObject() && dst.isMember(key) && dst[key].isObject()) {
                jsonMergeArrays(src[key], dst[key]);
            } else {
                dst[key] = src[key];
            }
        }
    }

    namespace {
        struct IndirectComparer {
            bool operator()(const Json::Value* a, const Json::Value* b) const {
                return *a < *b;
            }
        };
    } // namespace

    Json::Value jsonRemoveArrayDuplicates(const Json::Value& src) {
        if (!src.isArray()) {
            return src;
        }

        Json::Value result(Json::arrayValue);
        std::set<const Json::Value*, IndirectComparer> cache;
        for (const auto& item : src) {
            auto [iter, inserted] = cache.emplace(&item);
            if (inserted) {
                result.append(item);
            }
        }
        return result;
    }

    Json::Value mapJsonArray(const Json::Value& value, std::function<Json::Value(const Json::Value&)> mapFunction) {
        if (!value.isArray()) {
            throw std::runtime_error("Value is not array");
        }
        Json::Value result(Json::arrayValue);
        for (const auto& item : value) {
            result.append(mapFunction(item));
        }
        return result;
    }

    /**
     * Convenience function to work with json configs.
     * Will throw a Json::exception if json is non-empty but invalid.
     *
     * Example (see testUtilsGetStringSafe) ::
     *
     *    getStringSafe(R"x(|{"foo": {"bar": "baz"}})x", ".foo.bar", "123"); // "baz"
     *    getStringSafe(R"x(|{"foo": {"bar": "baz"}})x", ".foo.pew", "123"); // "123");
     *    getStringSafe("", ".foo.bar", "123"); // "123"
     *
     */
    std::string getStringSafe(const std::string& probablyJson, const std::string& jsonPath, const std::string& defaultValue)
    {
        if (probablyJson.empty()) {
            return defaultValue;
        }

        auto json = parseJson(probablyJson);
        auto path = Json::Path(jsonPath);

        return path.resolve(json, Json::Value(defaultValue)).asString();
    }

    std::chrono::milliseconds tryGetMillis(const Json::Value& json, const std::string& name, std::chrono::milliseconds defaultValue)
    {
        return std::chrono::milliseconds(tryGetInt64(json, name, defaultValue.count()));
    }

    std::chrono::seconds tryGetSeconds(const Json::Value& json, const std::string& name, std::chrono::seconds defaultValue)
    {
        return std::chrono::seconds(tryGetInt64(json, name, defaultValue.count()));
    }
} // namespace quasar
