#pragma once

#include <json/json.h>

#include <util/generic/typetraits.h>

#include <chrono>
#include <concepts>
#include <functional>
#include <optional>
#include <set>
#include <string>
#include <string_view>
#include <unordered_set>

namespace quasar {
    std::optional<Json::Value> tryParseJson(std::string_view jsonStr);

    Json::Value tryParseJsonOrEmpty(std::string_view jsonStr);

    Json::Value parseJson(std::string_view jsonStr);

    Json::Value parseJson(std::istream& ifs);

    /// traverse entire json and transform each string with function f
    Json::Value transform(const Json::Value& value, std::function<std::string(const std::string&)> f);

    Json::Value readJsonFromFile(const std::string& filename);

    std::optional<Json::Value> tryReadJsonFromFile(const std::string& filename);

    Json::Value getConfigFromFile(const std::string& configFile);

    Json::Value tryGetConfigFromFile(const std::string& configFile, const Json::Value& defaultConfig);

    Json::Value getJson(const Json::Value& json, const std::string& field);

    Json::Value
    tryGetJson(const Json::Value& json, const std::string& field, const Json::Value& defaultValue = Json::Value());

    Json::Value
    tryGetArray(const Json::Value& json, const std::string& field, const Json::Value& defaultValue = Json::Value());

    Json::Value
    getArray(const Json::Value& json, const std::string& field);

    const Json::Value* findXPath(const Json::Value& json, const std::string& xpath);
    Json::Value* findXPath(Json::Value& json, const std::string& xpath);

    std::string getString(const Json::Value& json, const std::string& field);

    std::string tryGetString(const Json::Value& json, const std::string& field, std::string defaultValue = "");

    Json::Value getArrayElement(const Json::Value& array, int index);

    int getInt(const Json::Value& json, const std::string& field);

    int64_t getInt64(const Json::Value& json, const std::string& field);

    int64_t tryGetInt64(const Json::Value& json, const std::string& field, int64_t defaultValue);

    uint64_t getUInt64(const Json::Value& json, const std::string& field);

    uint64_t tryGetUInt64(const Json::Value& json, const std::string& field, uint64_t defaultValue);

    uint32_t getUInt32(const Json::Value& json, const std::string& field);

    uint32_t tryGetUInt32(const Json::Value& json, const std::string& field, uint32_t defaultValue);

    uint8_t tryGetUInt8(const Json::Value& json, const std::string& field, uint8_t defaultValue);

    int tryGetInt(const Json::Value& json, const std::string& field, int defaultValue);

    double getDouble(const Json::Value& json, const std::string& field);

    double tryGetDouble(const Json::Value& json, const std::string& field, double defaultValue);

    float tryGetFloat(const Json::Value& json, const std::string& field, float defaultValue);

    bool checkHasDoubleField(const Json::Value& json, const std::string& field);

    bool getBool(const Json::Value& json, const std::string& field);

    bool tryGetBool(const Json::Value& json, const std::string& field, bool defaultValue);

    std::optional<bool> getOptionalBool(const Json::Value& json, const std::string& field);

    Json::Value recursiveExpandJson(Json::Value src);

    std::string jsonToString(const Json::Value& json, bool omitEndingLineFeed = false);

    /**
     * Merges json objects without intersection of Json::Array's
     * Arrays in @param dst will be just replaced by arrays from @param src
     **/
    void jsonMerge(const Json::Value& src, Json::Value& dst);

    /**
     * Merges json objects including Json::Array's merge
     * Arrays from @param src will be merged into arrays of @param dst
     **/
    void jsonMergeArrays(const Json::Value& src, Json::Value& dst);

    /**
     * Removes duplicate items from json array
     */
    Json::Value jsonRemoveArrayDuplicates(const Json::Value& src);

    std::string
    getStringSafe(const std::string& probablyJson, const std::string& jsonPath, const std::string& defaultValue = "");

    /**
     * Applies provided function to array members.
     * @param value input JSON value. Must be array.
     * @param mapFunction function which is applied to array members.
     * @return result
     * @throws std::runtime_error if object is not array.
     * @throws every exception mapFunction throws
     */
    Json::Value mapJsonArray(const Json::Value& value, std::function<Json::Value(const Json::Value&)> mapFunction);

    template <typename Vec_>
    Json::Value vectorToJson(const Vec_& src) {
        Json::Value result(Json::arrayValue);
        for (auto& val : src) {
            result.append(val);
        }
        return result;
    }

    template <typename T>
    concept jsonDeserializable = requires(T object, const Json::Value& json) {
        object.parseJson(json);
    };

    template <typename T>
    std::vector<T> jsonToVec(const Json::Value& json, const std::vector<T>& defaultValue = std::vector<T>{}) {
        if (!json.isArray()) {
            return defaultValue;
        }

        std::vector<T> result;
        result.reserve(json.size());

        for (const auto& element : json) {
            if (element.isNull()) {
                continue;
            }
            try {
                if constexpr (std::is_same_v<bool, T>) {
                    result.push_back(element.asBool());
                } else if constexpr (std::is_unsigned_v<T>) {
                    result.push_back(static_cast<T>(element.asUInt64()));
                } else if constexpr (std::is_signed_v<T>) {
                    result.push_back(static_cast<T>(element.asInt64()));
                } else if constexpr (std::is_floating_point_v<T>) {
                    result.push_back(static_cast<T>(element.asDouble()));
                } else if constexpr (std::is_same_v<std::string, T>) {
                    result.push_back(element.asString());
                } else if constexpr (jsonDeserializable<T> && std::is_default_constructible<T>::value && std::movable<T>) {
                    T elem;
                    elem.parseJson(element);
                    result.push_back(std::move(elem));
                } else {
                    static_assert(TDependentFalse<T>);
                }
            } catch (...) {
                return defaultValue;
            }
        }

        return result;
    }

    template <typename T>
    std::set<T> jsonToSet(const Json::Value& json) {
        std::set<T> set;
        for (auto& value : jsonToVec<T>(json)) {
            set.insert(std::move(value));
        }

        return set;
    }

    template <class T>
    std::vector<T> tryGetVector(const Json::Value& json, const std::string& field, const std::vector<T>& defaultValue = std::vector<T>{}) {
        return jsonToVec<T>(tryGetArray(json, field), defaultValue);
    }

    std::chrono::milliseconds tryGetMillis(const Json::Value& json, const std::string& name, std::chrono::milliseconds defaultValue);
    std::chrono::seconds tryGetSeconds(const Json::Value& json, const std::string& name, std::chrono::seconds defaultValue);

    template <class T>
    concept stringEmplaceable = requires(T container, std::string str) {
        container.emplace(std::move(str));
    };
    template <class T>
    requires stringEmplaceable<T>
        T tryGetEmplaceableStringSet(const Json::Value& json, const std::string& fieldName, const T& defaultValue) {
        const auto& field = json[fieldName];
        if (field.isString()) {
            return {field.asString()};
        }

        try {
            if (field.isArray()) {
                T value;
                for (auto& v : field) {
                    value.emplace(v.asString());
                }
                return value;
            }
        } catch (...) {
        }

        try {
            if (field.isObject()) {
                T value;
                for (const auto& key : field.getMemberNames()) {
                    if (!field[key].isBool()) {
                        throw std::exception();
                    }
                    if (field[key].asBool()) {
                        value.emplace(key);
                    }
                }
                return value;
            }
        } catch (...) {
        }

        return defaultValue;
    }
} // namespace quasar
