#include "jwt.h"

#include <yandex_io/libs/errno/errno_exception.h>

#include <cerrno>

using namespace quasar;

JwtPtr quasar::decodeJWT(const std::string& token, const std::string& key) {
    JwtPtr jwt;
    jwt_t* t;
    int error = jwt_decode(
        &t, token.c_str(), (key.empty() ? nullptr : (const unsigned char*)key.c_str()), key.size());
    if (error) {
        throw std::runtime_error(
            "Can't parse jwt \"" + token + "\" with key \"" + key + "\". Error: " + strError(error));
    }
    jwt.reset(t);
    return jwt;
}

std::string quasar::encodeJWT(const std::unordered_map<std::string, JwtValue>& values, jwt_alg_t alg, const std::string& key) {
    JwtPtr jwt;
    jwt_t* t;
    jwt_new(&t);
    jwt.reset(t);

    for (const auto& pair : values)
    {
        switch (pair.second.getType()) {
            case JwtValue::Type::LONG:
                jwt_add_grant_int(jwt.get(), pair.first.c_str(), pair.second.asLong());
                break;
            case JwtValue::Type::STRING:
                jwt_add_grant(jwt.get(), pair.first.c_str(), pair.second.asString().c_str());
                break;
        }
    }

    jwt_set_alg(jwt.get(), alg, (key.empty() ? nullptr : (const unsigned char*)key.c_str()), key.size());

    auto encodedJwt = JwtStringPtr(jwt_encode_str(jwt.get()));
    return std::string(encodedJwt.get());
}

long quasar::getLongGrantFromJWT(jwt_t* token, const std::string& grantName) {
    auto result = jwt_get_grant_int(token, grantName.c_str());
    if (errno == ENOENT) {
        throw std::runtime_error("grant \"" + grantName + "\" is not found or is not an integer");
    }
    return result;
}

bool quasar::getBoolOrFalseGrantFromJWT(jwt_t* token, const std::string& grantName) {
    auto result = jwt_get_grant_bool(token, grantName.c_str());
    if (errno == ENOENT) {
        return false;
    }
    return result;
}

std::string quasar::getStringGrantFromJWT(jwt_t* token, const std::string& grantName) {
    auto result = jwt_get_grant(token, grantName.c_str());
    if (result == nullptr) {
        throw std::runtime_error("grant \"" + grantName + "\" is not found or is not a string");
    }
    return std::string(result);
}
