#include "message_packer.h"

#include <util/system/byteorder.h>

#include <arpa/inet.h>

using namespace quasar;
using namespace quasar::keymaster_proxy_client;

// NOLINTNEXTLINE(readability-convert-member-functions-to-static)
std::vector<char> MessagePacker::pack(const std::string& msg) const {
    if (msg.length() > MAX_SIZE) {
        throw std::runtime_error("Maximum size of request message is exceeded (" + std::to_string(msg.length()) + " > " + std::to_string(MAX_SIZE) + ")");
    }

    std::vector<char> bytes(msg.length() + 2 * sizeof(uint32_t));

    uint32_t bigEndianMagic = HostToInet<uint32_t>(MAGIC);
    std::memcpy(bytes.begin(), &bigEndianMagic, sizeof(uint32_t));

    uint32_t bigEndianLength = HostToInet<uint32_t>(msg.length());
    std::memcpy(bytes.begin() + sizeof(uint32_t), &bigEndianLength, sizeof(uint32_t));

    std::memcpy(bytes.begin() + 2 * sizeof(uint32_t), msg.data(), msg.length());

    return bytes;
}

// NOLINTNEXTLINE(readability-convert-member-functions-to-static)
std::string MessagePacker::unpack(SignerTransportBase& transport) const {
    const std::vector<char> magicBytes = transport.receive(sizeof(uint32_t));
    uint32_t bigEndianMagic;
    memcpy(&bigEndianMagic, magicBytes.begin(), sizeof(uint32_t));
    const uint32_t magic = InetToHost<uint32_t>(bigEndianMagic);
    if (magic != MAGIC) {
        transport.close();
        throw std::runtime_error("Wrong magic number, connection is closed");
    }

    const std::vector<char> lengthBytes = transport.receive(sizeof(uint32_t));
    uint32_t bigEndianLength;
    memcpy(&bigEndianLength, lengthBytes.begin(), sizeof(uint32_t));
    const uint32_t length = InetToHost<uint32_t>(bigEndianLength);

    if (length > MAX_SIZE) {
        throw std::runtime_error("Maximum size of response payload is exceeded (" + std::to_string(length) + " > " + std::to_string(MAX_SIZE) + ")");
    }

    auto payloadBytes = transport.receive(length);
    return std::string(payloadBytes.begin(), payloadBytes.end());
}
