#include "kv_database.h"

#include <yandex_io/libs/logging/logging.h>

#include <contrib/libs/lmdbxx/lmdb++.h>

#include <util/folder/path.h>

YIO_DEFINE_LOG_MODULE("kv_database");

KVDatabase::KVDatabase(const std::string& path)
    : env_(std::make_unique<lmdb::env>(lmdb::env::create()))
{
    if (path.empty()) {
        YIO_LOG_ERROR_EVENT("KVDatabase.InitFailed", "Path is empty");

        env_.reset();

        return;
    }

    try {
        TFsPath(path).Parent().MkDirs();

        env_->set_mapsize(10 << 20); // 10 MiB
        env_->set_max_dbs(32);
        env_->open(path.c_str(), MDB_NOSUBDIR | MDB_NOLOCK);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("KVDatabase.InitFailed", "Failed to init database: " << e.what());

        env_.reset();
    }
}

KVDatabase::~KVDatabase() = default;

void KVDatabase::store(const std::string& db, const std::string& key, const std::string& value) {
    std::scoped_lock lock(mutex_);

    if (!env_) {
        return;
    }

    try {
        auto txn = lmdb::txn::begin(*env_);
        auto dbi = lmdb::dbi::open(txn, db.c_str(), MDB_CREATE);

        dbi.put(txn, key, value);

        txn.commit();
    } catch (const lmdb::error& e) {
        YIO_LOG_ERROR_EVENT("KVDatabase.StoreFailed", "Failed to store key: " << e.what());
    }
}

std::optional<std::string> KVDatabase::load(const std::string& db, const std::string& key) {
    std::scoped_lock lock(mutex_);

    if (!env_) {
        return std::nullopt;
    }

    try {
        auto txn = lmdb::txn::begin(*env_);
        auto dbi = lmdb::dbi::open(txn, db.c_str(), MDB_CREATE);

        std::string_view value;

        if (dbi.get(txn, key, value)) {
            return std::string{value};
        }
    } catch (const lmdb::error& e) {
        YIO_LOG_ERROR_EVENT("KVDatabase.LoadFailed", "Failed to load key: " << e.what());
    }

    return std::nullopt;
}

void KVDatabase::erase(const std::string& db, const std::string& key) {
    std::scoped_lock lock(mutex_);

    if (!env_) {
        return;
    }

    try {
        auto txn = lmdb::txn::begin(*env_);
        auto dbi = lmdb::dbi::open(txn, db.c_str(), MDB_CREATE);

        dbi.del(txn, key);
    } catch (const lmdb::error& e) {
        YIO_LOG_ERROR_EVENT("KVDatabase.EraseFailed", "Failed to erase key: " << e.what());
    }
}

std::map<std::string, std::string> KVDatabase::loadAll(const std::string& db) {
    std::scoped_lock lock(mutex_);

    if (!env_) {
        return {};
    }

    try {
        auto txn = lmdb::txn::begin(*env_);
        auto dbi = lmdb::dbi::open(txn, db.c_str(), MDB_CREATE);
        auto cursor = lmdb::cursor::open(txn, dbi);

        std::map<std::string, std::string> values;

        std::string_view key;
        std::string_view value;

        if (cursor.get(key, value, MDB_FIRST)) {
            do {
                values.emplace(std::string{key}, std::string{value});
            } while (cursor.get(key, value, MDB_NEXT));
        }

        return values;
    } catch (const lmdb::error& e) {
        YIO_LOG_ERROR_EVENT("KVDatabase.LoadFailed", "Failed to load DB: " << e.what());
    }

    return {};
}

void KVDatabase::eraseAll(const std::string& db) {
    std::scoped_lock lock(mutex_);

    if (!env_) {
        return;
    }

    try {
        auto txn = lmdb::txn::begin(*env_);
        auto dbi = lmdb::dbi::open(txn, db.c_str(), MDB_CREATE);

        dbi.drop(txn);
    } catch (const lmdb::error& e) {
        YIO_LOG_ERROR_EVENT("KVDatabase.EraseFailed", "Failed to erase DB: " << e.what());
    }
}
