#pragma once

#include "lmdb_helpers.h"

#include <optional>

namespace quasar {

    template <typename Result, typename Callback, typename KeyType>
    struct applyToBoolImpl {
        static bool apply(Callback& cb, const KeyType& key, const std::string_view& value) {
            return cb(key, value);
        }
    };

    template <typename Callback, typename KeyType>
    struct applyToBoolImpl<void, Callback, KeyType> {
        static bool apply(Callback& cb, const KeyType& key, const std::string_view& value) {
            cb(key, value);
            return true;
        }
    };

    template <typename KeysConv>
    class LmdbDbi {
        lmdb::dbi dbi_;

    public:
        using KeyType = typename KeysConv::KeyType;
        struct KeyValue {
            KeyType key;
            std::string_view value;
        };

        template <typename Callback>
        bool applyToBool(Callback& cb, const KeyType& key, const std::string_view& value) {
            using ResultType = std::invoke_result_t<Callback, const KeyType&, const std::string_view&>;
            return applyToBoolImpl<ResultType, Callback, KeyType>::apply(cb, key, value);
        }

        class Cursor {
            lmdb::cursor cursor_;

            std::optional<KeyType> get(MDB_cursor_op mode) {
                std::string_view key;
                if (cursor_.get(key, mode)) {
                    return KeysConv::fromSv(key);
                }
                return std::nullopt;
            }

            std::optional<KeyValue> getKeyVal(MDB_cursor_op mode) {
                std::string_view key;
                std::string_view value;
                if (cursor_.get(key, value, mode)) {
                    return KeyValue{.key = KeysConv::fromSv(key), .value = value};
                }
                return std::nullopt;
            }

        public:
            Cursor(lmdb::txn& txn, LmdbDbi& dbi)
                : cursor_(lmdb::cursor::open(txn, dbi.dbi_))
            {
            }

            std::optional<KeyType> first() {
                return get(MDB_FIRST);
            }

            std::optional<KeyType> last() {
                return get(MDB_LAST);
            }

            std::optional<KeyType> next() {
                return get(MDB_NEXT);
            }

            std::optional<KeyType> prev() {
                return get(MDB_PREV);
            }

            std::optional<KeyValue> firstKeyVal() {
                return getKeyVal(MDB_FIRST);
            }
            std::optional<KeyValue> lastKeyVal() {
                return getKeyVal(MDB_LAST);
            }
            std::optional<KeyValue> nextKeyVal() {
                return getKeyVal(MDB_NEXT);
            }
            std::optional<KeyValue> prevKeyVal() {
                return getKeyVal(MDB_PREV);
            }

            std::optional<KeyType> toKey(const KeyType& srcKey) {
                auto dbKey = KeysConv::toDbKey(srcKey);
                std::string_view key = lmdb::to_sv(dbKey);
                if (cursor_.get(key, MDB_SET_RANGE)) {
                    return KeysConv::fromSv(key);
                }
                return std::nullopt;
            }

            std::optional<KeyValue> toKeyVal(const KeyType& srcKey) {
                KeyValue result{
                    .key = KeysConv::toDbKey(srcKey),
                };
                std::string_view key = lmdb::to_sv(result.key);
                if (cursor_.get(key, result.value, MDB_SET_RANGE)) {
                    result.key = KeysConv::fromSv(key);
                    return result;
                }
                return std::nullopt;
            }

            void del() {
                cursor_.del();
            }

            void replace(const KeyType& srcKey, std::string_view value) {
                auto dbKey = KeysConv::toDbKey(srcKey);
                cursor_.put(lmdb::to_sv(dbKey), value, MDB_CURRENT);
            }
        };

        LmdbDbi() = default;

        LmdbDbi(LmdbDbi&& src) noexcept
            : dbi_(std::move(src.dbi_))
        {
        }

        LmdbDbi(lmdb::env& env, const char* name)
            : dbi_(util::initDbi(env, name))
        {
        }

        LmdbDbi& operator=(LmdbDbi&& src) noexcept {
            dbi_ = std::move(src.dbi_);
            return *this;
        }

        Cursor openCursor(lmdb::txn& txn) {
            return Cursor(txn, *this);
        }

        void forEach(lmdb::txn& txn, auto callback) {
            auto cursor = openCursor(txn);
            if (auto keyVal = cursor.firstKeyVal()) {
                do {
                    if (!applyToBool(callback, keyVal->key, keyVal->value)) {
                        return;
                    };
                } while (keyVal = cursor.nextKeyVal());
            }
        }

        void forEachReverse(lmdb::txn& txn, auto callback) {
            auto cursor = openCursor(txn);
            if (auto keyVal = cursor.lastKeyVal()) {
                do {
                    if (!applyToBool(callback, keyVal->key, keyVal->value)) {
                        return;
                    };
                } while (keyVal = cursor.prevKeyVal());
            }
        }

        MDB_stat stat(lmdb::txn& txn) {
            return dbi_.stat(txn);
        }

        void put(lmdb::txn& txn, const KeyType& key, std::string_view value) {
            KeyType dbKey = KeysConv::toDbKey(key);
            dbi_.put(txn, lmdb::to_sv(dbKey), value);
        }

        void remove(const auto& keys, lmdb::env& env, auto onRemove) {
            for (auto idx : keys) {
                auto txn = lmdb::txn::begin(env);
                {
                    auto cursor = openCursor(txn);
                    auto dbKey = KeysConv::indexToKey(idx);
                    if (auto keyVal = cursor.toKeyVal(dbKey)) {
                        if (KeysConv::equal(keyVal->key, idx)) {
                            onRemove(keyVal->value);
                            cursor.del();
                        }
                    }
                }
                txn.commit();
            }
        }
    };

} // namespace quasar
