#include "helpers.h"

#include <util/string/ascii.h>

#include <sstream>
#include <string>

#include <charconv>

spdlog::level::level_enum quasar::Logging::helpers::getLogLevelFromString(std::string_view logLevel) {
    struct LevelName {
        std::string_view name;
        spdlog::level::level_enum level;
    };

    static constexpr auto levels = std::array<LevelName, 6>{{
        {"trace", spdlog::level::trace},
        {"debug", spdlog::level::debug},
        {"info", spdlog::level::info},
        {"warn", spdlog::level::warn},
        {"error", spdlog::level::err},
        {"fatal", spdlog::level::critical},
    }};

    auto logLevelLower = std::string(logLevel);
    for (auto& c : logLevelLower) {
        c = AsciiToLower(c);
    }

    for (const auto& i : levels) {
        if (i.name == logLevelLower) {
            return i.level;
        }
    }

    std::ostringstream message;
    message << "Unknown logging level: '" << logLevel << "'.";
    message << " Available levels: ";
    bool first = true;
    for (const auto& i : levels) {
        if (first) {
            first = false;
        } else {
            message << ", ";
        }
        message << i.name;
    }
    throw std::runtime_error(message.str());
}

size_t quasar::Logging::helpers::parseFileSize(std::string_view humanSize) {
    size_t multiplier = 1;

    const size_t bytesIndex = humanSize.find_first_of("bB");
    if (bytesIndex != std::string_view::npos && bytesIndex > 0) {
        humanSize.remove_suffix(humanSize.size() - bytesIndex);

        const char c = humanSize.back();
        if (c == 'k' || c == 'K') {
            multiplier = 1024;
        } else if (c == 'm' || c == 'M') {
            multiplier = 1024 * 1024;
        } else if (c == 'g' || c == 'G') {
            multiplier = 1024 * 1024 * 1024;
        }

        if (multiplier != 1) {
            humanSize.remove_suffix(1);
        }
    }

    size_t unitCount = 0;
    auto parseResult = std::from_chars(humanSize.begin(), humanSize.end(), unitCount);
    if (parseResult.ec != std::errc() || parseResult.ptr != humanSize.end()) {
        throw std::runtime_error("Invalid file size: " + std::string(humanSize));
    }

    return unitCount * multiplier;
}
