#pragma once

// TODO get rid of this header
#include <contrib/restricted/spdlog/include/spdlog/spdlog.h>

#include <util/system/compiler.h>

#include <sstream>

namespace quasar::Logging {

    class Logger {
    public:
        Logger();

        explicit operator bool() const;

        bool shouldLog(spdlog::level::level_enum level) const;

        void log(spdlog::source_loc loc, spdlog::level::level_enum level, const std::string& msg);

    private:
        std::shared_ptr<spdlog::logger> logger_;
    };

} // namespace quasar::Logging

// NOTE: This is not the most efficient implementation:
// 1. The std::shared_ptr version of default_logger() is used, i.e. there is reference
// counting on each logging call. Reference counting may be dropped if spdlog
// initialization is guaranteed to happen in main() before any multi-threaded
// logging calls.
// 2. std::ostringstream is not optimized for repeated re-allocation of the same
// buffer (it could be cached in a thread_local: this would trade memory for
// logging overhead)
// 3. Low-probability logging invocation could be placed in an outline function
// to reduce logging footprint on useful code size.
#define YIO_SPDLOG_LOGGER_CALL(level, message)                                                                  \
    if (auto __logger = ::quasar::Logging::Logger(); bool(__logger) && __logger.shouldLog(level)) {             \
        std::ostringstream __spdlog_ostream;                                                                    \
        __spdlog_ostream << message;                                                                            \
        __logger.log(::spdlog::source_loc{__FILE__, __LINE__, SPDLOG_FUNCTION}, level, __spdlog_ostream.str()); \
    }

#define YIO_SPDLOG_IS_LOG_LEVEL_ENABLED(level) ([]() -> bool { \
    auto __logger = ::quasar::Logging::Logger();               \
    return bool(__logger) && __logger.shouldLog(level);        \
}())

// Implementation detail: current logging module
#define YIO_LOG_MODULE_VAR static_log_module_var__use_YIO_DEFINE_LOG_MODULE_macro_if_not_defined__

// Annotate subsequent logging calls as belonging to a certain logical module.
// * YIO_DEFINE_LOG_MODULE is a top-level statement for the entire compilation
//   unit (cannot be used in headers).
// * YIO_DEFINE_LOG_MODULE_IN_SCOPE is a local statement for inline functions in
//   headers (e.g. template functions)
#define YIO_DEFINE_LOG_MODULE(module)                                       \
    namespace {                                                             \
        [[maybe_unused]] constexpr const char* YIO_LOG_MODULE_VAR = module; \
    }
#define YIO_DEFINE_LOG_MODULE_IN_SCOPE(module) static constexpr const char* YIO_LOG_MODULE_VAR = module

// Check whether a non-default log level is enabled.
// Computation of extra data for verbose logging level must be disabled
// by checking these conditions.
#define YIO_LOG_TRACE_ENABLED() Y_UNLIKELY(YIO_SPDLOG_IS_LOG_LEVEL_ENABLED(::spdlog::level::trace))
#define YIO_LOG_DEBUG_ENABLED() Y_UNLIKELY(YIO_SPDLOG_IS_LOG_LEVEL_ENABLED(::spdlog::level::debug))

// Simple ("textual") logging macros.
// `message` is a sequence of `a << b << c` ostream operators.
#define YIO_LOG_TRACE(message) YIO_SPDLOG_LOGGER_CALL(::spdlog::level::trace, message)
#define YIO_LOG_DEBUG(message) YIO_SPDLOG_LOGGER_CALL(::spdlog::level::debug, message)
#define YIO_LOG_INFO(message) YIO_SPDLOG_LOGGER_CALL(::spdlog::level::info, message)
#define YIO_LOG_WARN(message) YIO_SPDLOG_LOGGER_CALL(::spdlog::level::warn, message)

// Implementation detail: unique substring surrounding log event metadata
#define YIO_LOG_EVENT_MARKER "@@LOG_EVENT@@"

#define YIO_BUILD_LOG_EVENT_HEADER_IMPL(CONCAT, logModuleName, logEventName) \
    YIO_LOG_EVENT_MARKER                                                     \
    "{\"module_name\":\"" CONCAT logModuleName CONCAT "\""                   \
    ",\"event_name\":\"" CONCAT logEventName CONCAT "\"}" YIO_LOG_EVENT_MARKER " "

// Escape JSON for fmt::format()
#define YIO_BUILD_LOG_EVENT_HEADER_FMT_IMPL(CONCAT, logModuleName, logEventName) \
    YIO_LOG_EVENT_MARKER                                                         \
    "{{\"module_name\":\"" CONCAT logModuleName CONCAT "\""                      \
    ",\"event_name\":\"" CONCAT logEventName CONCAT "\"}}" YIO_LOG_EVENT_MARKER " "

// Event-generating logging macros. These logging calls provide named events for monitoring.
// `message` is a sequence of `a << b << c` ostream operators.
// `logEventName` must be a static compile-time string, that is not changed in
// later versions (while message formatting may be changed, event name must
// persist as long as its semantics are unchanged).
#define YIO_LOG_ERROR_EVENT(logEventName, message) YIO_SPDLOG_LOGGER_CALL( \
    ::spdlog::level::err,                                                  \
    YIO_BUILD_LOG_EVENT_HEADER_IMPL(<<, YIO_LOG_MODULE_VAR, logEventName)  \
        << message)

// #define YIO_LOG_DEBUG_ENABLED() true
// #define YIO_LOG_TRACE_ENABLED() false
// #define YIO_LOG_TRACE(message) (std::cout << message << std::endl)
// #define YIO_LOG_DEBUG(message) (std::cout << message << std::endl)
// #define YIO_LOG_INFO(message) (std::cout << message << std::endl)
// #define YIO_LOG_WARN(message) (std::cout << message << std::endl)
// #define YIO_LOG_ERROR(message) (std::cout << message << std::endl)
