#include "setup.h"

#include <yandex_io/libs/logging/helpers.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/logging/sinks/ipc/ipc_sink.h>
#include <yandex_io/libs/logging/sinks/rotating_file/rotating_file_sink.h>
#include <yandex_io/libs/logging/sinks/telemetry/telemetry_sink.h>

#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>

#include <contrib/restricted/spdlog/include/spdlog/spdlog.h>
#include <contrib/restricted/spdlog/include/spdlog/sinks/stdout_color_sinks.h>

#include <json/json.h>

#ifdef __ANDROID__
    #include <contrib/restricted/spdlog/include/spdlog/sinks/android_sink.h>
#endif

#include <string>
#include <string_view>

namespace {

    void configureSpdlogDefaultLogger(std::string_view logLevelString, std::vector<std::shared_ptr<spdlog::sinks::sink>> sinks, std::string_view pattern = quasar::Logging::DEFAULT_PATTERN) {
        auto logLevel = quasar::Logging::helpers::getLogLevelFromString(logLevelString);
        auto logger = std::make_shared<spdlog::logger>("");
        logger->set_level(logLevel);

        // Flush every log record
        logger->flush_on(logLevel);

        // NOTE: spdlog logger configuration is not thread-safe
        // all configuration should be completed *BEFORE* installing this logger
        logger->sinks() = std::move(sinks);

        // NOTE: This effectively sets pattern on all sinks, thus must be called after
        // setup function
        logger->set_pattern(std::string{pattern});

        spdlog::set_default_logger(std::move(logger));
    }

} // anonymous namespace

void quasar::Logging::initLoggingToStdout(const std::string& logLevel, std::string_view pattern) {
    configureSpdlogDefaultLogger(logLevel, {std::make_shared<spdlog::sinks::stdout_color_sink_mt>()}, pattern);
}

void quasar::Logging::initLoggingToFile(const std::string& logFileName, const std::string& logLevel, size_t maxFileSize, size_t maxBackupIndex) {
    auto fileSink = std::make_shared<spdlog_compat::sinks::rotating_file_sink_mt>(
        /* base_filename = */ logFileName,
        /* max_size = */ maxFileSize,
        /* max_files = */ maxBackupIndex,
        /* rotate_on_open = */ true);
    configureSpdlogDefaultLogger(logLevel, {fileSink});
}

void quasar::Logging::initLogging(const Json::Value& configuration) {
    const auto& loggingConfig = configuration["Logging"];
    std::string logLevel = getString(loggingConfig, "level");
    std::vector<std::shared_ptr<spdlog::sinks::sink>> sinks;

    if (loggingConfig.isMember("file") && tryGetBool(loggingConfig["file"], "enabled", false)) {
        const auto& fileConfig = loggingConfig["file"];
        std::string logFileName = getString(fileConfig, "fileName");
        size_t maxFileSize = helpers::parseFileSize(getString(fileConfig, "maxFileSize"));
        size_t maxBackupIndex = getInt(fileConfig, "maxBackupIndex");

        auto fileSink = std::make_shared<spdlog_compat::sinks::rotating_file_sink_mt>(
            /* base_filename = */ std::move(logFileName),
            /* max_size = */ maxFileSize,
            /* max_files = */ maxBackupIndex,
            /* rotate_on_open = */ true);
        sinks.push_back(std::move(fileSink));
    }

    if (loggingConfig.isMember("stdout") && tryGetBool(loggingConfig["stdout"], "enabled", false)) {
        auto stdoutSink = std::make_shared<spdlog::sinks::stdout_color_sink_mt>();
        sinks.push_back(std::move(stdoutSink));
    }

#ifdef __ANDROID__
    if (loggingConfig.isMember("android") && tryGetBool(loggingConfig["android"], "enabled", false)) {
        const auto& androidConfig = loggingConfig["android"];
        std::string logTag = getString(androidConfig, "tag");
        auto androidSink = std::make_shared<spdlog::sinks::android_sink_mt>(std::move(logTag));
        sinks.push_back(std::move(androidSink));
    }
#endif

    configureSpdlogDefaultLogger(logLevel, std::move(sinks));
}

void quasar::Logging::addLoggingToTelemetryIfNeeded(const Json::Value& configuration, std::shared_ptr<YandexIO::ITelemetry> telemetry) {
    Y_UNUSED(configuration);

    auto telemetrySink = std::make_shared<TelemetrySink>(std::move(telemetry));
    telemetrySink->set_level(spdlog::level::err);
    addLoggingSink(std::move(telemetrySink));
}

void quasar::Logging::addLoggingToIpcIfNeeded(const Json::Value& configuration, std::shared_ptr<ipc::IIpcFactory> ipcFactory) {
    const auto& loggingConfig = configuration["Logging"];
    if (!loggingConfig.isMember("ipc") || !tryGetBool(loggingConfig["ipc"], "enabled", false)) {
        return;
    }

    const auto& ipcConfig = loggingConfig["ipc"];
    std::string serviceName = getString(ipcConfig, "serviceName");

    auto ipcSink = std::make_shared<IpcSink>(ipcFactory, serviceName);
    addLoggingSink(std::move(ipcSink));
}

void quasar::Logging::addLoggingSink(std::shared_ptr<spdlog::sinks::sink> loggingSink) {
    try {
        auto oldLogger = spdlog::default_logger();
        auto newLogger = std::make_shared<spdlog::logger>(*oldLogger);
        newLogger->sinks().push_back(std::move(loggingSink));
        spdlog::set_default_logger(std::move(newLogger));
    } catch (const std::exception& e) {
        // TODO: do something
    }
}

void quasar::Logging::changeLoggerLevel(const std::string& level) {
    try {
        ::spdlog::default_logger()->set_level(quasar::Logging::helpers::getLogLevelFromString(level));
    } catch (const std::exception& e) {
        // TODO: do something
    }
}

void quasar::Logging::deinitLogging() {
    spdlog::shutdown();
}
