#pragma once

#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/telemetry/telemetry.h>

#include <contrib/restricted/spdlog/include/spdlog/sinks/sink.h>

#include <string>

namespace Json {
    class Value;
} // namespace Json

/**
 * Logging subsystem start/stop helpers.
 *
 * `initLogging*()` functions are used to prepare the logging subsystem.
 *
 * NB:
 * - Only one of these functions is supposed to be called in a program. A
 *   subsequent call would reset the logging subsystem according to the new
 *   configuration.
 * - Logging should be initialized in main() as early as possible before
 *   spawning threads and writing to log.
 */

namespace quasar::Logging {

    // Prepend "[PID:TID]" to default log format (yandex_io code is multi-threaded)
    inline constexpr std::string_view DEFAULT_PATTERN = "[%P:%t] %+";

    /**
     * @brief Config-less setup for console logging.
     */
    void initLoggingToStdout(const std::string& logLevel, std::string_view pattern = DEFAULT_PATTERN);

    /**
     * @brief Config-less setup for file logging.
     */
    void initLoggingToFile(const std::string& logFileName, const std::string& logLevel, size_t maxFileSize, size_t maxBackupIndex);

    /**
     * @brief Setup logging configuration from YandexIO config.
     *
     * Installs a common production set of loggers:
     * - rotating file (enabled by \p file key)
     * - stdout logger (enabled by \p stdout key)
     * - Android logger (enabled by \p android key)
     */
    void initLogging(const Json::Value& configuration);

    /**
     * @brief Setup additional logging to telemetry.
     */
    void addLoggingToTelemetryIfNeeded(const Json::Value& configuration, std::shared_ptr<YandexIO::ITelemetry> telemetry);

    /**
     * @brief Setup additional logging to an IPC service.
     */
    void addLoggingToIpcIfNeeded(const Json::Value& configuration, std::shared_ptr<ipc::IIpcFactory> ipcFactory);

    /**
     * Add logging sink at runtime
     * @param loggingSink
     */
    void addLoggingSink(std::shared_ptr<spdlog::sinks::sink> loggingSink);

    /**
     * Changes logging level at runtime
     * @param level
     */
    void changeLoggerLevel(const std::string& level);

    /**
     * @brief Stop logging subsystem.
     *
     * All loggers are destroyed, all background threads are stopped.
     * Logging calls after \p deinitLogging() are a programming error and will
     * likely result in a crash.
     */
    void deinitLogging();

} // namespace quasar::Logging
