#include "ipc_sink.h"

#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar::Logging;

namespace {

    thread_local bool logging_ = false;

} // namespace

IpcSink::IpcSink(const std::shared_ptr<ipc::IIpcFactory>& ipcFactory, const std::string& serviceName)
    : connector_(ipcFactory->createIpcConnector(serviceName))
{
    connector_->connectToService();
}

void IpcSink::sink_it_(const spdlog::details::log_msg& msg) {
    if (logging_) {
        // Prevent recursive messages from IPC stack
        return;
    }

    logging_ = true;

    spdlog::memory_buf_t formatted;
    formatter_->format(msg, formatted);

    auto message = ipc::buildMessage([&](auto& message) {
        auto* log = message.mutable_log_message();

        log->set_timestamp_ms(std::chrono::duration_cast<std::chrono::milliseconds>(msg.time.time_since_epoch()).count());
        log->set_msg(std::string{formatted.data(), formatted.size()});

        auto* source = log->mutable_source();
        source->set_filename(msg.source.filename);
        source->set_line(msg.source.line);
        source->set_funcname(msg.source.funcname);

        switch (msg.level) {
            case spdlog::level::trace:
                log->set_level(proto::LogMessage::TRACE);
                break;
            case spdlog::level::debug:
                log->set_level(proto::LogMessage::DEBUG);
                break;
            case spdlog::level::info:
                log->set_level(proto::LogMessage::INFO);
                break;
            case spdlog::level::warn:
                log->set_level(proto::LogMessage::WARN);
                break;
            case spdlog::level::err:
                log->set_level(proto::LogMessage::ERROR);
                break;
            default:
                break;
        }
    });

    connector_->sendMessage(message);

    logging_ = false;
}

void IpcSink::flush_() {
}
