#include "logging_handler.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/logging/setup/setup.h>

#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;

static const std::string LOGGING = "Logging";

LoggingHandler::~LoggingHandler() {
    std::lock_guard guard(mutex_);
    if (started_) {
        started_ = false;
        toSync_->shutdown();
        quasar::Logging::deinitLogging();
    }
}

void LoggingHandler::init(const std::shared_ptr<ipc::IIpcFactory>& ipcFactory, const std::string& serviceName, const Json::Value& quasarConfig, std::shared_ptr<YandexIO::ITelemetry> telemetry) {
    std::lock_guard<std::mutex> guard(mutex_);
    if (started_) {
        throw std::runtime_error("LoggingHandler already started");
    }
    if (!quasarConfig.isMember(serviceName)) {
        throw std::runtime_error("Service \"" + serviceName + "\" not found in quasar config. Fail to create LoggingHandler.");
    }

    ipcFactory_ = ipcFactory;
    serviceName_ = serviceName;
    patchFilename_ = getString(quasarConfig["common"], "dataDir") + "/logging/" + serviceName_ + ".cfg";
    telemetry_ = std::move(telemetry);
    originalConfig_[LOGGING] = quasarConfig[serviceName_][LOGGING];

    reinitLoggingSubsystem();

    toSync_ = ipcFactory_->createIpcConnector("syncd");
    toSync_->setMessageHandler(std::bind(&LoggingHandler::onQuasarMessage, this, std::placeholders::_1));
    toSync_->connectToService();

    started_ = true;
}

void LoggingHandler::reinitLoggingSubsystem() {
    auto finalConfig = originalConfig_;
    if (auto patch = tryReadJsonFromFile(patchFilename_.GetPath())) {
        if (patch->isMember(LOGGING)) {
            jsonMerge(*patch, finalConfig);
        }
    }

    if (finalConfig_ == finalConfig) {
        return;
    }

    if (started_) {
        YIO_LOG_INFO("Shutdown logging subsytem before reinitializing...");
    }

    finalConfig_ = std::move(finalConfig);

    Logging::initLogging(finalConfig_);
    if (telemetry_) {
        Logging::addLoggingToTelemetryIfNeeded(finalConfig_, telemetry_);
    }
    if (ipcFactory_) {
        Logging::addLoggingToIpcIfNeeded(finalConfig_, ipcFactory_);
    }
    bool isPatched = (originalConfig_ != finalConfig_);
    YIO_LOG_INFO("Initialize logging subsystem " << (isPatched ? "(patched) " : "") << ": " << jsonToString(finalConfig_[LOGGING], true));
}

void LoggingHandler::onQuasarMessage(const ipc::SharedMessage& message) {
    if (message->has_user_config_update() && message->user_config_update().has_config()) {
        Json::Value loggingPatch = Json::nullValue;

        const auto& config = parseJson(message->user_config_update().config());
        if (!config["system_config"].isNull()) {
            const auto& systemConfig = getJson(config, "system_config");
            if (systemConfig[serviceName_].isObject()) {
                if (systemConfig[serviceName_][LOGGING].isObject()) {
                    loggingPatch[LOGGING] = systemConfig[serviceName_][LOGGING];
                }
            }
            if (systemConfig["logging"].isObject() && systemConfig["logging"][serviceName_].isObject() && systemConfig["logging"][serviceName_]["level"].isString()) {
                loggingPatch[LOGGING]["level"] = systemConfig["logging"][serviceName_]["level"].asString();
            }
        }

        {
            std::lock_guard<std::mutex> guard(mutex_);
            if (started_) {
                if (loggingPatch.isNull()) {
                    patchFilename_.ForceDelete();
                } else {
                    patchFilename_.Parent().MkDirs();
                    TransactionFile file(patchFilename_.GetPath());
                    file.write(jsonToString(loggingPatch));
                    file.commit();
                }
                reinitLoggingSubsystem();
            }
        }
    }
}
