#include "client_stats.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

using namespace quasar;

namespace {
    template <typename ProtoSource>
    std::vector<proto::MetricaMessage::StatsRecord> toVector(const ProtoSource& source) {
        std::vector<proto::MetricaMessage::StatsRecord> result;
        for (auto item : source) {
            result.push_back(item);
        }
        return result;
    }
} // namespace

ClientStats::ClientStats(OnStats onStats)
    : onStats_(std::move(onStats))
{
    if (!onStats_) {
        throw std::runtime_error("ClientStats doesnt work without stats handler");
    }
}

ClientStats::~ClientStats() {
    periodicExecutor_.reset();
}

void ClientStats::processStatsMessage(const Statistics& stats) {
    std::scoped_lock<std::mutex> lock(statsMutex_);
    if (stats.has_service_name()) {
        auto dataVec = toVector(stats.data());
        auto [iter, inserted] = clientStats_.emplace(stats.service_name(), dataVec);
        if (!inserted) {
            iter->second = std::move(dataVec);
        }
    }
}

Json::Value ClientStats::statisticsEventPayload() {
    StatsMap stats;
    {
        std::scoped_lock<std::mutex> lock(statsMutex_);
        stats.swap(clientStats_);
    }
    Json::Value result = Json::objectValue;
    if (stats.empty()) {
        return result;
    }

    // FIXME: should telemetry be protobufs?
    for (const auto& [serviceName, dataSrc] : stats) {
        Json::Value data = Json::objectValue;
        for (const auto& dataItem : dataSrc) {
            Json::Value counters = Json::arrayValue;
            for (const auto cnt : dataItem.counters()) {
                counters.append(cnt);
            }
            data[dataItem.name()] = std::move(counters);
        }
        result[serviceName] = std::move(data);
    }
    return result;
}

void ClientStats::applyConfig(const Json::Value& cfg) {
    if (const auto newPeriod = tryGetInt(cfg, "statsPeriodSec", 0); newPeriod > 0) {
        YIO_LOG_DEBUG("telemetry clients stats period changed to " << newPeriod << " seconds");
        if (periodicExecutor_) {
            periodicExecutor_->setPeriodTime(std::chrono::seconds(newPeriod));
        } else {
            periodicExecutor_ = std::make_unique<PeriodicExecutor>(
                [this]() {
                    auto eventValue = statisticsEventPayload();
                    if (!eventValue.empty()) {
                        onStats_(std::move(eventValue));
                    }
                },
                std::chrono::seconds(newPeriod),
                PeriodicExecutor::PeriodicType::SLEEP_FIRST);
        }
    } else {
        YIO_LOG_DEBUG("telemetry clients stats disabled");
        periodicExecutor_.reset();
    }
}
