#include "events_database.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/json.h>
#include <yandex_io/libs/sqlite/sqlite_database.h>

#include <util/generic/scope.h>
#include <util/system/yassert.h>

#include <memory>
#include <stdexcept>
#include <unordered_set>

#include <sys/stat.h>
#include <random>

YIO_DEFINE_LOG_MODULE("metrica_base");

using namespace quasar;
using namespace quasar::proto;

namespace quasar {
    static std::ostream& operator<<(std::ostream& os, const quasar::EventsDatabase::DbParams& dbParams) {
        return os << dbParams.path;
    }

    std::string rndMark4() {
        return rndMark(4);
    }
} // namespace quasar

class EventsDatabase::DbSql: public SqliteDatabase {
public:
    DbSql(EventsDatabase::DbParams param)
        : SqliteDatabase(param.path, param.maxSizeKB)
        , dbParam(std::move(param))
        , bootMark([&] {
            std::string mark;
            if (dbParam.markAdd && !dbParam.markBootPrefixPath.empty()) {
                try {
                    mark = getFileContent(dbParam.markBootPrefixPath);
                } catch (...) {
                }
                if (mark.empty() || mark.length() != 4) {
                    mark = rndMark4();
                    try {
                        PersistentFile pf(dbParam.markBootPrefixPath, PersistentFile::Mode::TRUNCATE);
                        pf.write(mark);
                    } catch (...) {
                        mark.clear();
                    }
                }
            }
            return mark;
        }())
    {
        YIO_LOG_DEBUG("EventsDatabase::DbSql [" << dbParam.path << "], "
                                                << "markAdd=" << (dbParam.markAdd ? "true" : "false")
                                                << ", bootPrefixPath=" << dbParam.markBootPrefixPath
                                                << ", bootMark=" << bootMark);
    }

    using SqliteDatabase::isEnoughMemoryForInsert;
    using SqliteDatabase::runQueryWithoutCallback;
    using SqliteDatabase::vacuum;

    std::string makeQueryError(std::string_view sql, std::string_view message) {
        SqliteDatabase::logQueryError(sql, message);
        return makeString("Error when initializing tables for database {", getDbFilename(), "}, query: ", sql, ", error: ", message);
    }

    sqlite3* sqlite() {
        return db_;
    }

    const EventsDatabase::DbParams dbParam;
    const std::string bootMark;
};

EventsDatabase::EventsDatabase(std::string databasePath, uint64_t maxSizeKB)
    : EventsDatabase(DbParams{std::move(databasePath), static_cast<int64_t>(maxSizeKB), nullptr, false, ""})
{
}

EventsDatabase::EventsDatabase(DbParams runtimeDb)
    : EventsDatabase(std::vector<DbParams>{std::move(runtimeDb)})
{
}

EventsDatabase::EventsDatabase(DbParams runtimeDb, DbParams persistentDb)
    : EventsDatabase(std::vector<DbParams>{std::move(runtimeDb), std::move(persistentDb)})
{
}

EventsDatabase::EventsDatabase(std::vector<DbParams> dbParams)
    : instanceMark_(rndMark4())
{
    if (dbParams.empty()) {
        std::string message = "Database list is empty";
        YIO_LOG_ERROR_EVENT("EventsDatabase.InitScheme.Exception", message);
        throw std::runtime_error(message);
    }

    constexpr const char* sqlCreateTable = "CREATE TABLE IF NOT EXISTS EVENTS(ID INTEGER PRIMARY KEY, SERIALIZED_DATA BLOB)";
    constexpr const char* sqlMaxId = "SELECT CASE WHEN MAX(ID) IS NULL THEN 0 ELSE MAX(ID) END FROM EVENTS";
    auto readId = [](void* pkptr, int /*arg_num*/, char** argv, char** /*col_names*/) -> int {
        auto& pk = *static_cast<uint64_t*>(pkptr);
        pk = std::max<uint64_t>((argv && argv[0] ? strtoul(argv[0], nullptr, 10) : 0), pk);
        return 0;
    };
    char* errmsg = nullptr;
    Y_DEFER {
        sqlite3_free(errmsg);
    };

    for (auto& dbParam : dbParams) {
        try {
            dbSqls_.push_back(std::make_shared<DbSql>(dbParam));
        } catch (const std::exception& ex) {
            YIO_LOG_ERROR_EVENT("EventsDatabase.InitScheme.Exception", "Failed to initialize events database: " << dbParam.path << ": " << ex.what());
            throw;
        }
        auto& dbSql = dbSqls_.back();
        if (sqlite3_exec(dbSql->sqlite(), sqlCreateTable, nullptr, nullptr, &errmsg) != SQLITE_OK) {
            throw std::runtime_error(dbSql->makeQueryError(sqlCreateTable, errmsg));
        }
        if (sqlite3_exec(dbSql->sqlite(), sqlMaxId, readId, &pk_, &errmsg) != SQLITE_OK) {
            throw std::runtime_error(dbSql->makeQueryError(sqlMaxId, errmsg));
        }
    }
    YIO_LOG_INFO("Open event database: PK=" << pk_ << ", mark=" << instanceMark_ << ", files=[" << join(dbParams, ", ") << "]");
}

/**
 * @brief Get earliest event from database, but it doesn't count events with specified ids
 * @param skipIds -- we get the earliest event from all but events with these id
 * @return unique_ptr to event, nullptr if there's no such event
 */
std::unique_ptr<EventsDatabase::Event> EventsDatabase::getEarliestEvent(const std::vector<uint64_t>& skipIds) {
    // TODO[labudidabudai] по-хорошему здесь нужно не в цикле отбирать события, а одним sql-запросом отбирать сначала
    // самое раннее событие вида "изменилось окружение", а потом все события до этого, будет гораздо быстрее
    std::lock_guard lock(mutex_);
    const std::string sqlFilter = (skipIds.empty() ? "WHERE TRUE" : makeString("WHERE ID NOT IN (", join(skipIds, ","), ")"));
    const std::string sqlMinId = "SELECT CASE WHEN MIN(ID) IS NULL THEN 0 ELSE MIN(ID) END FROM EVENTS " + sqlFilter;
    uint64_t minId = 0;
    DbSql* minDbSql = nullptr;
    for (auto& dbSql : dbSqls_) {
        uint64_t id{0};
        auto readId = [](void* idptr, int /*arg_num*/, char** argv, char** /*col_names*/) -> int {
            auto& id = *static_cast<uint64_t*>(idptr);
            id = (argv && argv[0] ? strtoul(argv[0], nullptr, 10) : 0);
            return 0;
        };
        char* errmsg = nullptr;
        Y_DEFER {
            sqlite3_free(errmsg);
        };
        if (sqlite3_exec(dbSql->sqlite(), sqlMinId.c_str(), readId, &id, &errmsg) != SQLITE_OK) {
            dbSql->makeQueryError(sqlMinId, errmsg);
            continue;
        }
        if (id && (id < minId || minId == 0)) {
            minId = id;
            minDbSql = dbSql.get();
        }
    }

    std::unique_ptr<EventsDatabase::Event> res = nullptr;
    if (!minDbSql) {
        return res;
    }

    auto callback = [](void* eventPointer, int numCols, char** colValues, char** colNames) -> int {
        uint64_t id = 0;
        uint64_t dataLength = 0;
        DatabaseMetricaEvent event;
        char* startEventData = nullptr;
        for (int i = 0; i < numCols; ++i) {
            if (!strcmp(colNames[i], "ID")) {
                std::stringstream stream;
                stream << colValues[i];
                stream >> id;
            } else if (!strcmp(colNames[i], "SERIALIZED_DATA")) {
                startEventData = colValues[i];
            } else if (!strcmp(colNames[i], "DATA_LENGTH")) {
                std::stringstream stream;
                stream << colValues[i];
                stream >> dataLength;
            }
        }
        if (startEventData && dataLength) {
            const TString eventData(startEventData, dataLength);
            if (!event.ParseFromString(eventData)) {
                YIO_LOG_ERROR_EVENT("EventsDatabase.GetEarliestEvent.ProtobufParseError", "Cannot parse event from database: " << eventData);
            }
        }
        *static_cast<std::unique_ptr<EventsDatabase::Event>*>(eventPointer) = std::make_unique<EventsDatabase::Event>(id, event);
        return 0;
    };
    const std::string query = makeString("SELECT *, LENGTH(SERIALIZED_DATA) AS DATA_LENGTH FROM EVENTS ", sqlFilter, " AND ID=", minId);
    char* errmsg = nullptr;
    Y_DEFER {
        sqlite3_free(errmsg);
    };
    if (sqlite3_exec(minDbSql->sqlite(), query.c_str(), callback, &res, &errmsg) != SQLITE_OK) {
        minDbSql->makeQueryError(query, errmsg);
    } else {
        if (minDbSql->dbParam.markAdd && res && res->databaseMetricaEvent.has_new_event()) {
            if (auto json = tryParseJson(res->databaseMetricaEvent.new_event().value())) {
                (*json)["_mrk"] = makeString(instanceMark_, ":", tryGetString(*json, "_mrk"));
                res->databaseMetricaEvent.mutable_new_event()->set_value(jsonToString(*json, true));
            }
        }
    }
    return res;
}

uint64_t EventsDatabase::addEvent(const DatabaseMetricaEvent& event) {
    std::lock_guard lock(mutex_);
    uint64_t newId = ++pk_;
    std::string query = makeString("INSERT INTO EVENTS(ID, SERIALIZED_DATA) VALUES (", newId, ", ?)");
    bool fLostEvent = false;
    bool fSuccess = false;
    for (auto& dbSql : dbSqls_) {
        const auto& dbParam = dbSql->dbParam;
        if (dbParam.filter && !dbParam.filter(newId, event, fSuccess)) {
            continue;
        }

        std::string buffer;
        if (dbSql->dbParam.markAdd && event.has_new_event()) {
            if (auto json = tryParseJson(event.new_event().value())) {
                (*json)["_mrk"] = makeString(instanceMark_, '.', dbSql->bootMark);
                auto mrkEvent = event;
                mrkEvent.mutable_new_event()->set_value(jsonToString(*json, true));
                buffer = mrkEvent.SerializeAsString();
            }
        }

        if (buffer.empty()) {
            buffer = event.SerializeAsString();
        }

        if (!dbSql->isEnoughMemoryForInsert(buffer)) {
            fLostEvent = true;
            continue;
        }

        // Here we execute statement in three steps:
        // 1. Preparing query, creating statement object
        // 2. Binding data to statement (place binary data to question sign in query)
        // 3. Stepping (executing)
        // It's done here that way because we have to bind binary data, so use only string in query in not good

        sqlite3_stmt* statement = nullptr;
        Y_DEFER {
            sqlite3_finalize(statement);
        };

        const auto& dbPath = dbSql->getDbFilename();
        auto* db = dbSql->sqlite();
        if (sqlite3_prepare_v2(db, query.c_str(), query.size(), &statement, nullptr) != SQLITE_OK) {
            YIO_LOG_ERROR_EVENT("EventsDatabase.SqliteError", "Error when preparing query " << query << ", database {" << dbPath << "}, error message: " << sqlite3_errmsg(db));
            continue;
        }

        if (sqlite3_bind_blob(statement, 1, buffer.c_str(), buffer.size(), SQLITE_STATIC) != SQLITE_OK) {
            YIO_LOG_ERROR_EVENT("EventsDatabase.SqliteError", "Error when binding blob to query " << query << ", database {" << dbPath << "}, error message: " << sqlite3_errmsg(db));
            continue;
        }

        if (sqlite3_step(statement) != SQLITE_DONE) {
            YIO_LOG_ERROR_EVENT("EventsDatabase.SqliteError", "Error when executing query " << query << ", database {" << dbPath << "}, error message: " << sqlite3_errmsg(db));
            continue;
        }

        fSuccess = true;
        YIO_LOG_DEBUG("EventsDatabase [" << dbParam.path << "] add event [" << newId << "]: " << convertMessageToDeepJsonString(event));
    }

    if (fSuccess) {
        if (lostEvents_) {
            YIO_LOG_INFO("There's now enough memory for new events, lost " << lostEvents_ << " events");
            lostEvents_ = 0;
        }
    } else if (fLostEvent) {
        YIO_LOG_WARN("Is not enough memory to write new event: " << event.new_event().name());
        ++lostEvents_;
    }
    return (fSuccess ? newId : 0);
}

EventsDatabase::MPResult EventsDatabase::makePersistent(const Filter& filter, uint64_t fromId) {
    if (!filter) {
        return {};
    }

    if (dbSqls_.size() == 1) {
        return {};
    }

    std::lock_guard lock(mutex_);
    auto runtimeDb = dbSqls_.front();
    auto persistentDbSql = dbSqls_.back();
    std::vector<uint64_t> excludeIds;
    excludeIds.reserve(1024);
    const std::string sqlAlreadyPersistent = makeString("SELECT ID FROM EVENTS WHERE ID >= ", fromId, " ORDER BY ID");
    auto readPersistentId = [](void* excludeIdsPtr, int /*arg_num*/, char** argv, char** /*col_names*/) -> int {
        auto& excludeIds = *static_cast<std::vector<uint64_t>*>(excludeIdsPtr);
        if (auto id = (argv && argv[0] ? strtoul(argv[0], nullptr, 10) : 0)) {
            excludeIds.push_back(id);
        }
        return 0;
    };

    char* errmsg = nullptr;
    Y_DEFER {
        sqlite3_free(errmsg);
    };
    if (sqlite3_exec(persistentDbSql->sqlite(), sqlAlreadyPersistent.c_str(), readPersistentId, &excludeIds, &errmsg) != SQLITE_OK) {
        persistentDbSql->makeQueryError(sqlAlreadyPersistent, errmsg);
        return {};
    }

    struct DupCtx {
        const std::string& rtdb;
        const std::string& psdb;
        const std::string& instanceMark;
        DbSql* dbSql;
        sqlite3* db;
        const Filter& filter;
        MPResult result;
    };
    auto duplicateCallback = [](void* dupCtxPtr, int argc, char** argv, char** names) -> int {
        std::string buffer;
        uint64_t eventId = 0;
        uint64_t dataLength = 0;
        char* startEventData = nullptr;
        for (int i = 0; i < argc; ++i) {
            if (!strcmp(names[i], "ID")) {
                std::stringstream stream;
                stream << argv[i];
                stream >> eventId;
            } else if (!strcmp(names[i], "SERIALIZED_DATA")) {
                startEventData = argv[i];
            } else if (!strcmp(names[i], "DATA_LENGTH")) {
                std::stringstream stream;
                stream << argv[i];
                stream >> dataLength;
            }
        }
        auto* dupCtx = static_cast<DupCtx*>(dupCtxPtr);
        dupCtx->result.eventMaxId = std::max(dupCtx->result.eventMaxId, eventId);

        if (startEventData && dataLength) {
            const TString eventData(startEventData, dataLength);
            DatabaseMetricaEvent event;
            if (!event.ParseFromString(eventData)) {
                YIO_LOG_ERROR_EVENT("EventsDatabase.MakePersistent.ProtobufParseError", "Cannot parse event from database: " << eventData);
            } else {
                try {
                    if (dupCtx->filter(eventId, event, false)) {
                        if (dupCtx->dbSql->dbParam.markAdd && event.has_new_event()) {
                            if (auto json = tryParseJson(event.mutable_new_event()->value())) {
                                if (!json->isMember("_mrk")) {
                                    (*json)["_mrk"] = makeString(dupCtx->instanceMark, '.', dupCtx->dbSql->bootMark);
                                    event.mutable_new_event()->set_value(jsonToString(*json, true));
                                    buffer = event.SerializeAsString();
                                    startEventData = (char*)buffer.data();
                                    dataLength = buffer.size();
                                }
                            }
                        }
                        std::string sqlInsert = makeString("INSERT INTO EVENTS(ID, SERIALIZED_DATA) VALUES (", eventId, ", ?)");
                        sqlite3_stmt* statement = nullptr;
                        Y_DEFER {
                            sqlite3_finalize(statement);
                        };

                        if (sqlite3_prepare_v2(dupCtx->db, sqlInsert.c_str(), sqlInsert.size(), &statement, nullptr) != SQLITE_OK) {
                            YIO_LOG_ERROR_EVENT("EventsDatabase.MakePersistent.SqliteError", "Fail mark event as persistance. Error when preparing query \"" << sqlInsert << "\": " << sqlite3_errmsg(dupCtx->db));
                            return 0;
                        }

                        if (sqlite3_bind_blob(statement, 1, startEventData, dataLength, SQLITE_STATIC) != SQLITE_OK) {
                            YIO_LOG_ERROR_EVENT("EventsDatabase.MakePersistent.SqliteError", "Fail mark event as persistance. Error when binding blob to query \"" << sqlInsert << "\": " << sqlite3_errmsg(dupCtx->db));
                            return 0;
                        }

                        if (sqlite3_step(statement) != SQLITE_DONE) {
                            YIO_LOG_ERROR_EVENT("EventsDatabase.MakePersistent.SqliteError", "Fail mark event as persistance. Error when executing query \"" << sqlInsert << "\": " << sqlite3_errmsg(dupCtx->db));
                            return 0;
                        }
                        YIO_LOG_DEBUG("EventsDatabase [" << dupCtx->rtdb << "->" << dupCtx->psdb << "] make persistent [" << eventId << "]: " << convertMessageToDeepJsonString(event));
                        dupCtx->result.eventCount += 1;
                        dupCtx->result.eventBytes += dataLength;
                    }
                } catch (const std::exception& ex) {
                    YIO_LOG_ERROR_EVENT("EventsDatabase.MakePersistent.Filter", "Exception: " << ex.what());
                } catch (...) {
                    YIO_LOG_ERROR_EVENT("EventsDatabase.MakePersistent.Filter", "Unexpected exception");
                }
            }
        }
        return 0;
    };

    const char* sqlHead = "SELECT ID, SERIALIZED_DATA, LENGTH(SERIALIZED_DATA) AS DATA_LENGTH FROM EVENTS ";
    auto sqlLowerBound = makeString("ID >= ", fromId, " ");
    auto sqlExclude = makeString("NOT (ID IN (", join(excludeIds), ")) ");
    auto sqlSelectEvent = makeString(sqlHead, "WHERE ", sqlLowerBound, " AND ", sqlExclude, " ORDER BY ID");
    DupCtx dupCtx{runtimeDb->getDbFilename(), persistentDbSql->getDbFilename(), instanceMark_, persistentDbSql.get(), persistentDbSql->sqlite(), filter, {}};
    if (sqlite3_exec(runtimeDb->sqlite(), sqlSelectEvent.c_str(), duplicateCallback, &dupCtx, &errmsg) != SQLITE_OK) {
        runtimeDb->makeQueryError(sqlSelectEvent, errmsg);
    }
    return dupCtx.result;
}

void EventsDatabase::deleteEvents(const std::vector<uint64_t>& ids, bool useVacuum) {
    if (ids.empty()) {
        return;
    }

    std::lock_guard lock(mutex_);
    YIO_LOG_DEBUG("EventsDatabase [" << join(dbSqls_, ",", [](const auto& v) { return v->getDbFilename(); }) << "] delete events [" << join(ids, ",") << "], useVacuum=" << useVacuum);
    const std::string queryStat = makeString("SELECT CASE WHEN MAX(ID) IS NULL THEN 0 ELSE MAX(ID) END, COUNT(*) FROM EVENTS WHERE ID IN (", join(ids, ","), ")");
    const std::string queryDelete = makeString("DELETE FROM EVENTS WHERE ID IN (", join(ids, ","), ")");
    for (auto& dbSql : dbSqls_) {
        struct Stat {
            uint64_t maxId = 0;
            uint64_t count = 0;
        } stat;
        auto getStat = [](void* statptr, int /*arg_num*/, char** argv, char** /*col_names*/) -> int {
            auto s = static_cast<Stat*>(statptr);
            s->maxId = (argv && argv[0] ? strtoul(argv[0], nullptr, 10) : 0);
            s->count = (argv && argv[1] ? strtoul(argv[1], nullptr, 10) : 0);
            return 0;
        };
        char* errmsg = nullptr;
        Y_DEFER {
            sqlite3_free(errmsg);
        };
        if (sqlite3_exec(dbSql->sqlite(), queryStat.c_str(), getStat, &stat, &errmsg) != SQLITE_OK) {
            YIO_LOG_WARN("EventsDatabase [" << dbSql->getDbFilename() << "] delete events failed: " << dbSql->makeQueryError(queryStat, errmsg));
            continue;
        }
        if (stat.count == 0) {
            YIO_LOG_DEBUG("EventsDatabase [" << dbSql->getDbFilename() << "] Noting to delete");
            continue;
        } else {
            YIO_LOG_DEBUG("EventsDatabase [" << dbSql->getDbFilename() << "] Remove " << stat.count << " events, max event id is " << stat.maxId);
        }
        auto changesBefore = sqlite3_total_changes(dbSql->sqlite());
        dbSql->runQueryWithoutCallback(queryDelete);
        auto changesAfter = sqlite3_total_changes(dbSql->sqlite());
        if (useVacuum && changesAfter > changesBefore) {
            YIO_LOG_DEBUG("Vacuum " << dbSql->getDbFilename() << " (" << changesAfter - changesBefore << " changes)");
            dbSql->vacuum();
        }
    }
}

const std::string& EventsDatabase::getDbFilename() const {
    return dbSqls_.front()->dbParam.path;
}

uint64_t EventsDatabase::getDatabaseSizeInBytes() {
    uint64_t totalSize = 0;
    std::lock_guard lock(mutex_);
    for (auto& dbSql : dbSqls_) {
        auto dbSize = dbSql->getDatabaseSizeInBytes();
        if (dbSize >= 0) {
            totalSize += static_cast<uint64_t>(dbSize);
        }
    }
    return totalSize;
}
