#include "events_database_creator.h"

#include <unordered_set>

using namespace quasar;

namespace {
    struct DbParamsArgs {
        int64_t maxSizeKb;
        std::string filename;
        std::unordered_set<std::string> whitelist;
        std::unordered_set<std::string> blacklist;
        bool mark;
        std::string bootPrefixPath;
    };
    DbParamsArgs loadDbParamsArgs(const Json::Value& config, const std::string& sectionName) {
        const auto& section = config[sectionName];
        if (section.isNull()) {
            throw std::runtime_error("Fail to create EventsDatabase: config must have \"" + sectionName + "\" section");
        }
        auto maxSizeKb = getUInt64(section, "maxSizeKb");
        if (!maxSizeKb) {
            throw std::runtime_error("Fail to create EventsDatabase: " + sectionName + ".maxSizeKb is invalid");
        }
        auto filename = getString(section, "filename");
        if (filename.empty()) {
            throw std::runtime_error("Fail to create EventsDatabase: " + sectionName + ".filename is empty");
        }
        auto whitelist = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(section, "whitelist", {});
        auto blacklist = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(section, "blacklist", {});
        auto mark = tryGetBool(section, "mark", false);
        auto bootPrefixPath = tryGetString(section, "bootPrefixPath");

        return DbParamsArgs{
            .maxSizeKb = static_cast<int64_t>(maxSizeKb),
            .filename = std::move(filename),
            .whitelist = std::move(whitelist),
            .blacklist = std::move(blacklist),
            .mark = mark,
            .bootPrefixPath = std::move(bootPrefixPath)};
    }
} // namespace

std::shared_ptr<EventsDatabase> quasar::createEventsDatabase(const Json::Value& config, EventsDatabase::TernaryFilter persistentFilter) {
    if (config.isNull()) {
        throw std::runtime_error("Fail to create EventsDatabase from null config");
    }

    auto runtimeArgs = loadDbParamsArgs(config, "runtime");
    EventsDatabase::DbParams runtimeDb{
        .path = std::move(runtimeArgs.filename),
        .maxSizeKB = runtimeArgs.maxSizeKb,
        .filter = [blacklist{std::move(runtimeArgs.blacklist)}](uint64_t /*id*/, const proto::DatabaseMetricaEvent& event, bool /*saved*/) {
            if (event.has_new_event()) {
                if (blacklist.count(event.new_event().name())) {
                    return false;
                }
            }
            return true;
        },
        .markAdd = runtimeArgs.mark,
        .markBootPrefixPath = std::move(runtimeArgs.bootPrefixPath),
    };

    if (config.isMember("persistent")) {
        auto persistentArgs = loadDbParamsArgs(config, "persistent");
        EventsDatabase::DbParams persistentDb{
            .path = std::move(persistentArgs.filename),
            .maxSizeKB = persistentArgs.maxSizeKb,
            .filter =
                [whitelist{std::move(persistentArgs.whitelist)},
                 persistentFilter{std::move(persistentFilter)}](uint64_t eventId, const proto::DatabaseMetricaEvent& event, bool saved) {
                    if (event.has_new_event()) {
                        if (whitelist.count(event.new_event().name())) {
                            return true;
                        }
                    }
                    if (persistentFilter) {
                        if (auto res = persistentFilter(eventId, event, saved)) {
                            return *res;
                        }
                    }
                    return false;
                },
            .markAdd = persistentArgs.mark,
            .markBootPrefixPath = std::move(persistentArgs.bootPrefixPath),
        };
        return std::make_shared<EventsDatabase>(std::move(runtimeDb), std::move(persistentDb));
    } else {
        return std::make_shared<EventsDatabase>(std::move(runtimeDb));
    }
}
