#include "metrica_router.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/counters/json_utils.h>

using namespace quasar;
using quasar::proto::DatabaseMetricaEvent;

MetricaRouter::MetricaRouter(
    std::shared_ptr<MetricaBlockingQueue> immediateQueue,
    std::shared_ptr<EventsDatabase> dbQueue,
    std::shared_ptr<MetricaSessionProvider> sessionProvider)
    : immediateQueue_(std::move(immediateQueue))
    , dbQueue_(std::move(dbQueue))
    , sessionProvider_(std::move(sessionProvider))
    , enabled_(true)
{
}

MetricaRouter::~MetricaRouter() {
}

void MetricaRouter::setEventWhitelist(std::unordered_set<std::string> eventWhitelist) {
    std::unique_lock lock(eventListsMutex_);
    eventWhitelist_ = std::move(eventWhitelist);
}

void MetricaRouter::setEventBlacklist(std::unordered_set<std::string> eventBlacklist) {
    std::unique_lock lock(eventListsMutex_);
    eventBlacklist_ = std::move(eventBlacklist);
}

void MetricaRouter::processEvent(const std::string& eventName, const std::string& eventValue, bool skipDatabase) {
    auto session = sessionProvider_->getAndIncrementSession();

    DatabaseMetricaEvent event;
    auto newEvent = event.mutable_new_event();
    newEvent->set_type(DatabaseMetricaEvent::NewEvent::CLIENT);
    newEvent->set_timestamp(std::time(nullptr));
    newEvent->set_name(TString(eventName));
    newEvent->set_value(TString(eventValue));
    newEvent->set_session_id(session.id);
    newEvent->set_serial_number(session.eventNumber);
    newEvent->set_connection_type(connectionType_.load());

    if (skipDatabase) {
        processImmediateEvent(event);
    } else {
        processDbEvent(event);
    }
}

void MetricaRouter::processError(const std::string& errorName, const std::string& errorValue, bool skipDatabase) {
    auto session = sessionProvider_->getAndIncrementSession();

    DatabaseMetricaEvent event;
    auto newEvent = event.mutable_new_event();
    newEvent->set_type(DatabaseMetricaEvent::NewEvent::ERROR);
    newEvent->set_timestamp(std::time(nullptr));
    newEvent->set_name(TString(errorName));
    newEvent->set_value(TString(errorValue));
    newEvent->set_session_id(session.id);
    newEvent->set_serial_number(session.eventNumber);
    newEvent->set_connection_type(connectionType_.load());

    if (skipDatabase) {
        processImmediateEvent(event);
    } else {
        processDbEvent(event);
    }
}

void MetricaRouter::putEnvironmentVariable(const std::string& variableName, const std::string& variableValue) {
    std::lock_guard<std::mutex> lock(environmentMutex_);

    const auto iterator = environmentVariables_.find(variableName);
    if (iterator != environmentVariables_.end() && iterator->second == variableValue) {
        return;
    }
    environmentVariables_[variableName] = variableValue;
    processNewEnvironment();
}

void MetricaRouter::deleteEnvironmentVariable(const std::string& variableName) {
    std::lock_guard<std::mutex> lock(environmentMutex_);

    if (environmentVariables_.erase(variableName) > 0) {
        processNewEnvironment();
    }
}

void MetricaRouter::processImmediateEvent(DatabaseMetricaEvent& event) {
    if (!enabled_) {
        dailyStats_.increment({Counters::IMMEDIATE, Counters::NOTENABLED_DROP});
        return;
    }

    setEventEnvironment(event);
    if (shouldProcessEvent(event)) {
        if (!immediateQueue_->tryPush(event)) {
            YIO_LOG_WARN("Queue overflowed with messages. New event has been dropped.");
            dailyStats_.increment({Counters::IMMEDIATE, Counters::IMMEDIATE_DROP});
        } else {
            dailyStats_.increment(Counters::IMMEDIATE);
        }
    } else {
        dailyStats_.increment({Counters::IMMEDIATE, Counters::BLACKLISTED});
    }
}

void MetricaRouter::processDbEvent(const DatabaseMetricaEvent& event) {
    if (!enabled_) {
        dailyStats_.increment({Counters::OVERDB, Counters::NOTENABLED_DROP});
        return;
    }

    if (shouldProcessEvent(event)) {
        if (dbQueue_->addEvent(event)) {
            dailyStats_.increment(Counters::OVERDB);
        } else {
            dailyStats_.increment({Counters::OVERDB, Counters::DBFAIL_DROP});
        }
    } else {
        dailyStats_.increment({Counters::OVERDB, Counters::BLACKLISTED});
    }
}

void MetricaRouter::processNewEnvironment() {
    if (!enabled_) {
        return;
    }

    DatabaseMetricaEvent event;
    setEventEnvironment(event);
    dbQueue_->addEvent(event);
}

void MetricaRouter::setEventEnvironment(DatabaseMetricaEvent& event) {
    auto environmentValues = event.mutable_new_environment()->mutable_environment_values();
    environmentValues->clear();
    for (auto const& item : environmentVariables_) {
        (*environmentValues)[item.first] = item.second;
    }
}

bool MetricaRouter::shouldProcessEvent(const DatabaseMetricaEvent& event) {
    auto& eventName = event.new_event().name();

    std::shared_lock lock(eventListsMutex_);
    const bool isWhitelisted = eventWhitelist_.empty() || eventWhitelist_.find(eventName) != eventWhitelist_.end();
    const bool isBlacklisted = !eventBlacklist_.empty() && eventBlacklist_.find(eventName) != eventBlacklist_.end();

    return isWhitelisted && !isBlacklisted;
}

void MetricaRouter::setEnabled(bool value) {
    std::lock_guard<std::mutex> lock(environmentMutex_);

    enabled_ = value;
    processNewEnvironment();
}

void MetricaRouter::setConnectionType(quasar::proto::ConnectionType connectionType) {
    connectionType_.store(connectionType);
}

Json::Value MetricaRouter::getStatisticsPayload() {
    return EnumCountersToJson(dailyStats_.getUpdatedCounters(), {"immedaite", "overdb", "notenabled_drop", "immediate_drop", "dbfail_drop", "blacklist"});
}
