#pragma once

#include "events_database.h"
#include "metrica_queue.h"
#include "metrica_session_provider.h"

#include <yandex_io/libs/counters/default_daily_counter.h>

#include <mutex>
#include <shared_mutex>
#include <unordered_set>

namespace quasar {

    class MetricaRouter {
        using DatabaseMetricaEvent = quasar::proto::DatabaseMetricaEvent;

    public:
        MetricaRouter(
            std::shared_ptr<MetricaBlockingQueue> immediateQueue,
            std::shared_ptr<EventsDatabase> dbQueue,
            std::shared_ptr<MetricaSessionProvider> sessionProvider);

        ~MetricaRouter();

        void processEvent(const std::string& event, const std::string& eventValue, bool skipDatabase = false);
        void processError(const std::string& error, const std::string& errorValue, bool skipDatabase = false);

        void putEnvironmentVariable(const std::string& variableName, const std::string& variableValue);
        void deleteEnvironmentVariable(const std::string& variableName);

        void setEventWhitelist(std::unordered_set<std::string> eventWhitelist);
        void setEventBlacklist(std::unordered_set<std::string> eventBlacklist);

        /*
         * When disabled it keeps track of environment variables,
         * but doesnt write any events to database or event queue
         */
        void setEnabled(bool value);
        void setConnectionType(quasar::proto::ConnectionType connectionType);

        Json::Value getStatisticsPayload();

    private:
        void processImmediateEvent(DatabaseMetricaEvent& event);
        void processDbEvent(const DatabaseMetricaEvent& event);
        void processNewEnvironment();
        void setEventEnvironment(DatabaseMetricaEvent& event);

        bool shouldProcessEvent(const DatabaseMetricaEvent& event);

    private:
        std::map<std::string, std::string> environmentVariables_;

        std::shared_ptr<MetricaBlockingQueue> immediateQueue_;
        std::shared_ptr<quasar::EventsDatabase> dbQueue_;
        std::shared_ptr<MetricaSessionProvider> sessionProvider_;

        std::shared_mutex eventListsMutex_;
        std::unordered_set<std::string> eventWhitelist_;
        std::unordered_set<std::string> eventBlacklist_;

        std::mutex environmentMutex_;
        std::atomic_bool enabled_;
        std::atomic<proto::ConnectionType> connectionType_{proto::CONNECTION_TYPE_UNKNOWN};

        enum Counters {
            IMMEDIATE,
            OVERDB,
            NOTENABLED_DROP,
            IMMEDIATE_DROP,
            DBFAIL_DROP,
            BLACKLISTED,
            MAX_
        };

        DefaultDailyCounter<Counters> dailyStats_;
    };

} // namespace quasar
