#include "metrica_session_provider.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/folder/path.h>

#include <cstdlib>
#include <fstream>
#include <string>

YIO_DEFINE_LOG_MODULE("metrica_base");

namespace {

    uint64_t getNowInSeconds() {
        const auto now = std::chrono::system_clock::now();
        auto timeInSec = std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch()).count();
        return static_cast<uint64_t>(timeInSec);
    }

    constexpr uint64_t SECONDS_IN_WEEK = 604800;

} // namespace

MetricaSessionProvider::MetricaSessionProvider(std::string sessionIdPersistentPart,
                                               std::string sessionIdTemporaryPart)
    : sessionIdPersistentPart_(std::move(sessionIdPersistentPart))
    , sessionIdTemporaryPart_(std::move(sessionIdTemporaryPart))
{
    TFsPath(sessionIdPersistentPart_).Parent().MkDirs();
    TFsPath(sessionIdTemporaryPart_).Parent().MkDirs();
    generator_.seed(quasar::getNowTimestampMs());
}

MetricaSessionProvider::Session MetricaSessionProvider::getAndIncrementSession() {
    std::lock_guard<std::mutex> guard(sessionMutex_);
    // assume session is not outdated
    if (!isInitialized || getNowInSeconds() - currentSession_.startTime >= SECONDS_IN_WEEK) {
        initSession(isInitialized);
    }
    return {currentSession_.id, currentSession_.startTime, currentSession_.eventNumber++};
}

MetricaSessionProvider::Session MetricaSessionProvider::generateNewSession() {
    std::lock_guard<std::mutex> guard(sessionMutex_);
    initSession(isInitialized);
    return {currentSession_.id, currentSession_.startTime, currentSession_.eventNumber++};
}

void MetricaSessionProvider::initSession(bool resetTemporaryPart) {
    const uint64_t persistentPart = readSessionIdPart(sessionIdPersistentPart_);
    const uint64_t temporaryPart = readSessionIdPart(sessionIdTemporaryPart_);

    auto newParts = incrementSessionIdParts(persistentPart, temporaryPart, resetTemporaryPart);

    saveSessionIdPart(sessionIdPersistentPart_, persistentPart, newParts.first);
    saveSessionIdPart(sessionIdTemporaryPart_, temporaryPart, newParts.second);

    currentSession_ = {
        getSessionId(newParts.first, newParts.second),
        getNowInSeconds(),
        1,
    };
    YIO_LOG_INFO("New sessionId parts=[" << newParts.first << " " << newParts.second << "], new sessionId = " << currentSession_.id);

    isInitialized = true;
}

std::pair<uint64_t, uint64_t> MetricaSessionProvider::incrementSessionIdParts(uint64_t persistentPart,
                                                                              uint64_t temporaryPart,
                                                                              bool resetTemporaryPart) {
    if (resetTemporaryPart || temporaryPart == 0) {
        return {persistentPart + 1, 1};
    } else {
        return {persistentPart, temporaryPart + 1};
    }
}

uint64_t MetricaSessionProvider::readSessionIdPart(const std::string& path) {
    std::ifstream input(path);
    uint64_t sessionIdPart = 0;
    if (input.good()) {
        if (input >> sessionIdPart) {
            YIO_LOG_INFO("Read " << sessionIdPart << " from " << path);
        } else {
            std::string base64Contents;
            try {
                base64Contents = quasar::base64EncodeFile(path);
            } catch (const std::exception&) {
                /* We are already handling the read failure, so it's kinda expected that it cannot be read. */
            }

            struct stat pathStat;
            std::ostringstream statsStringStream;
            if (stat(path.c_str(), &pathStat) == 0) {
                statsStringStream << "size = " << pathStat.st_size;
                statsStringStream << ", mtime = " << pathStat.st_mtime;
                statsStringStream << ", perm = " << quasar::getPermissionsString(pathStat.st_mode);
            } else {
                statsStringStream << "Failed to read stats, errno = " << errno;
            }

            YIO_LOG_ERROR_EVENT("MetricaSessionProvider.FailedReadSessionId", "Failed to read sessionId part from " << path << ". "
                                                                                                                    << "eof: " << input.eof()
                                                                                                                    << ", bad: " << input.bad()
                                                                                                                    << ", fail: " << input.fail()
                                                                                                                    << ", Contents: \"" << base64Contents << "\""
                                                                                                                    << ", Stats: [" << statsStringStream.str() << "]");
        }
    } else {
        YIO_LOG_WARN(path << " doesn't exist");
    }
    return sessionIdPart;
}

void MetricaSessionProvider::saveSessionIdPart(const std::string& path, uint64_t oldSessionIdPart,
                                               uint64_t newSessionIdPart) {
    if (oldSessionIdPart == newSessionIdPart) {
        return;
    }

    quasar::TransactionFile file{path};
    try {
        if (!file.write(std::to_string(newSessionIdPart))) {
            YIO_LOG_ERROR_EVENT("MetricaSessionProvider.FailedSaveSessionIdWrite", "Failed to save sessionId part to temporary file");
            return;
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("MetricaSessionProvider.FailedSaveSessionIdException", "Failed to save sessionId part to temporary file. " << e.what());
        return;
    }

    if (!file.commit()) {
        YIO_LOG_ERROR_EVENT("MetricaSessionProvider.FailedSaveSessionIdCommit", "Failed to save sessionId part to " << path);
        return;
    }

    YIO_LOG_INFO("Saved " << newSessionIdPart << " to " << path);
}

uint64_t MetricaSessionProvider::getSessionId(uint64_t persistentPart, uint64_t temporaryPart) {
    return (persistentPart << 32) | temporaryPart;
}
