#include "utils.h"

#include <curl/curl.h>

#include <cstring>
#include <sstream>
#include <stdexcept>

std::string urlencode(const std::string& str) {
    std::string urlencoded;
    CURL* curl = curl_easy_init();

    if (curl) {
        char* output = curl_easy_escape(curl, str.c_str(), static_cast<int>(str.size()));
        if (output) {
            urlencoded = std::string(output);
            curl_free(output);
            curl_easy_cleanup(curl);
        } else {
            curl_easy_cleanup(curl);
            throw std::runtime_error("Unable to urlencode string: " + str);
        }
    } else {
        throw std::runtime_error("Unable to init curl");
    }
    return urlencoded;
}

std::string compress_string(const std::string& str, int compressionlevel)
{
    z_stream zs; // z_stream is zlib's control structure
    memset(&zs, 0, sizeof(zs));
    constexpr int ZLIB_GZIP_HEADER_OFFSET = 16;
    constexpr int ZLIB_MEMORY_USAGE_LEVEL = 1;
    if (deflateInit2(&zs, compressionlevel, Z_DEFLATED, 9 + ZLIB_GZIP_HEADER_OFFSET, ZLIB_MEMORY_USAGE_LEVEL, Z_DEFAULT_STRATEGY) != Z_OK) {
        throw(std::runtime_error("deflateInit2 failed while compressing."));
    }

    zs.next_in = (Bytef*)str.data();
    zs.avail_in = static_cast<uInt>(str.size()); // set the z_stream's input

    int ret;
    char outbuffer[204800]; // Should theoretically be enough to compress all the possible payload in one go
    std::string outstring;

    // retrieve the compressed bytes blockwise
    do {
        zs.next_out = reinterpret_cast<Bytef*>(outbuffer);
        zs.avail_out = sizeof(outbuffer);

        ret = deflate(&zs, Z_FINISH);

        if (outstring.size() < zs.total_out) {
            // append the block to the output string
            outstring.append(outbuffer, zs.total_out - outstring.size());
        }
    } while (ret == Z_OK);

    deflateEnd(&zs);

    if (ret != Z_STREAM_END) { // an error occurred that was not EOF
        std::ostringstream oss;
        oss << "Exception during zlib compression: (" << ret << ") " << zs.msg;
        throw(std::runtime_error(oss.str()));
    }
    return outstring;
}
