#include "clickdaemon_consumer.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/metrica/base/events_database_creator.h>

#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/split.hpp>

using namespace quasar;

const std::string ClickdaemonConsumer::CONSUMER_NAME = "clickdaemon";

ClickdaemonConsumer::ClickdaemonConsumer(const Json::Value& config, std::shared_ptr<YandexIO::IDevice> device)
    : telemetry_(device->telemetry())
{
    const auto immediateQueueSize = getUInt64(config, "immediateQueueSize");
    immediateQueue_ = std::make_unique<MetricaBlockingQueue>(immediateQueueSize);

    dbQueue_ = quasar::createEventsDatabase(config,
                                            [this](uint64_t /*id*/, const proto::DatabaseMetricaEvent& event, bool saved) -> std::optional<bool> {
                                                if (!saved && event.has_new_event()) {
                                                    auto importantEvents = (std::lock_guard(importantEventsMutex_), importantEvents_);
                                                    if (importantEvents && importantEvents->count(event.new_event().name())) {
                                                        return true;
                                                    }
                                                }
                                                return std::nullopt;
                                            });

    auto sessionProvider = std::make_unique<MetricaSessionProvider>(
        getString(config, "metricaSessionIdPersistentPart"),
        getString(config, "metricaSessionIdTemporaryPart"));

    metricaRouter_ = std::make_unique<MetricaRouter>(
        immediateQueue_, dbQueue_,
        std::move(sessionProvider));

    auto whitelist = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(config, "eventWhitelist", {});
    metricaRouter_->setEventWhitelist(std::move(whitelist));
    auto blacklist = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(config, "eventBlacklist", {});
    metricaRouter_->setEventBlacklist(std::move(blacklist));

    const auto endpointUri = getString(config, "endpointUri");
    const auto eventsBatchSize = getUInt64(config, "eventsBatchSize");
    const auto metadata = ClickdaemonMetadata::load(getString(config, "metricaMetadataPath"));
    metricaSender_ = std::make_unique<ClickdaemonSender>(
        endpointUri, eventsBatchSize,
        immediateQueue_, dbQueue_,
        metadata,
        std::move(device));

    auto envKeysBlacklist = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(config, "envKeysBlacklist", {});
    metricaSender_->setEnvKeysBlacklist(std::move(envKeysBlacklist));

    setEnabled(config["enabled"].asBool());

    const auto sendMetricaPeriod = std::chrono::milliseconds(config["sendMetricaPeriodMs"].asUInt64());
    sendMetricaExecutor_ = std::make_unique<PeriodicExecutor>(
        PeriodicExecutor::PECallback([this](PeriodicExecutor* executor) {
            sendMetrica(executor);
        }), sendMetricaPeriod, PeriodicExecutor::PeriodicType::CALLBACK_FIRST);

    const auto sendDbSizePeriod = std::chrono::milliseconds(config["sendDbSizePeriodMs"].asUInt64());
    sendDbSizeExecutor_ = std::make_unique<PeriodicExecutor>(
        std::bind(&ClickdaemonConsumer::sendDbSize, this),
        sendDbSizePeriod);

    // report that SessionId are ordered and can be used to sort metrics
    putEnvironmentVariable("session_id_ordered", "1");
}

ClickdaemonConsumer::~ClickdaemonConsumer() {
}

void ClickdaemonConsumer::processStats(Json::Value payload) {
    std::string strPayload;
    {
        Json::Value fullPayload = Json::objectValue;
        fullPayload["daemons"] = std::move(payload);
        fullPayload["router"] = metricaRouter_->getStatisticsPayload();
        fullPayload["sender"] = metricaSender_->getStatsPayload();
        strPayload = jsonToString(fullPayload);
    };
    YIO_LOG_INFO("clickdaemon telemetryStats " << strPayload);
    metricaRouter_->processEvent("telemetryStats", strPayload, false);
}

void ClickdaemonConsumer::processEvent(const std::string& event, const std::string& eventValue, bool skipDatabase) {
    metricaRouter_->processEvent(event, eventValue, skipDatabase);
}

void ClickdaemonConsumer::processError(const std::string& error, const std::string& errorValue, bool skipDatabase) {
    metricaRouter_->processError(error, errorValue, skipDatabase);
}

void ClickdaemonConsumer::putEnvironmentVariable(const std::string& variableName, const std::string& variableValue) {
    metricaRouter_->putEnvironmentVariable(variableName, variableValue);
}

void ClickdaemonConsumer::deleteEnvironmentVariable(const std::string& variableName) {
    metricaRouter_->deleteEnvironmentVariable(variableName);
}

void ClickdaemonConsumer::setConnectionType(proto::ConnectionType connectionType) {
    metricaRouter_->setConnectionType(connectionType);
}

const std::string& ClickdaemonConsumer::getName() const {
    return CONSUMER_NAME;
}

void ClickdaemonConsumer::processConfigUpdate(const Json::Value& configUpdate, const Json::Value& fullConfig) {
    YIO_LOG_INFO("Got config update: " << jsonToString(configUpdate));

    if (configUpdate.isMember("eventWhitelist")) {
        std::unordered_set<std::string> whitelist;

        if (configUpdate["eventWhitelist"].isString()) {
            const auto whitelistStr = tryGetString(configUpdate, "eventWhitelist");
            if (!whitelistStr.empty()) {
                boost::split(whitelist, whitelistStr, boost::is_any_of(","));
            }
        } else {
            whitelist = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(fullConfig, "eventWhitelist", {});
        }

        metricaRouter_->setEventWhitelist(std::move(whitelist));
    }

    if (configUpdate.isMember("eventBlacklist")) {
        std::unordered_set<std::string> blacklist;

        if (configUpdate["eventBlacklist"].isString()) {
            const auto blacklistStr = tryGetString(configUpdate, "eventBlacklist");
            if (!blacklistStr.empty()) {
                boost::split(blacklist, blacklistStr, boost::is_any_of(","));
            }
        } else {
            blacklist = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(fullConfig, "eventBlacklist", {});
        }

        metricaRouter_->setEventBlacklist(std::move(blacklist));
    }

    if (const auto* currentConfig = (configUpdate.isMember("eventImportantList") ? &configUpdate : &fullConfig)) {
        auto eventImportantList = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(*currentConfig, "eventImportantList", {});
        auto importantEvents = std::make_shared<decltype(eventImportantList)>(std::move(eventImportantList));
        YIO_LOG_DEBUG("Clickdaemon eventImportantList: [" << join(*importantEvents, ",") << "]");
        importantEvents_ = (std::lock_guard(importantEventsMutex_), importantEvents);
        metricaSender_->setEventImportantList(importantEvents);
    }

    if (configUpdate.isMember("endpointUri")) {
        const auto endpointUri = configUpdate["endpointUri"].asString();
        metricaSender_->setEndpointUri(endpointUri);
    }

    if (configUpdate.isMember("eventsBatchSize")) {
        const auto eventsBatchSize = configUpdate["eventsBatchSize"].asUInt64();
        metricaSender_->setEventsBatchSize(eventsBatchSize);
    }

    if (configUpdate.isMember("envKeysBlacklist")) {
        auto envKeysBlacklist = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(fullConfig, "envKeysBlacklist", {});
        metricaSender_->setEnvKeysBlacklist(std::move(envKeysBlacklist));
    }

    if (configUpdate.isMember("sendMetricaPeriodMs")) {
        const auto sendMetricaPeriod = std::chrono::milliseconds(configUpdate["sendMetricaPeriodMs"].asUInt64());
        sendMetricaExecutor_->setPeriodTime(sendMetricaPeriod);
    }

    if (configUpdate.isMember("sendDbSizePeriodMs")) {
        const auto sendDbSizePeriod = std::chrono::milliseconds(configUpdate["sendDbSizePeriodMs"].asUInt64());
        sendDbSizeExecutor_->setPeriodTime(sendDbSizePeriod);
    }

    if (configUpdate.isMember("enabled")) {
        setEnabled(configUpdate["enabled"].asBool());
    }
}

void ClickdaemonConsumer::sendMetrica(quasar::PeriodicExecutor* executor) {
    if (!enabled_) {
        return;
    }

    std::chrono::milliseconds maxOperationTime = executor->periodTime() / 2;
    std::chrono::milliseconds operationTime;
    auto operationBegin = std::chrono::steady_clock::now();
    size_t count = 0;
    size_t iterations = 0;
    size_t totalCount = 0;
    do {
        count = metricaSender_->flushEvents();
        if (!totalCount || count) {
            ++iterations;
        }
        totalCount += count;
        operationTime = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - operationBegin);
    } while (count && operationTime < maxOperationTime);

    if (totalCount) {
        YIO_LOG_DEBUG("Sent " << totalCount << " events to clickdaemon, operation time is " << operationTime.count() << " ms, for " << iterations << " iteration(s)");
    }

    std::lock_guard<std::mutex> guard(dbSizeMutex_);
    dbSize_.process(dbQueue_->getDatabaseSizeInBytes());
}

void ClickdaemonConsumer::sendDbSize() {
    if (!enabled_) {
        return;
    }

    Json::Value dbSizeData;
    {
        std::lock_guard<std::mutex> guard(dbSizeMutex_);
        dbSizeData["value"]["min"] = dbSize_.getMin();
        dbSizeData["value"]["max"] = dbSize_.getMax();
        dbSizeData["value"]["mean"] = dbSize_.getMean();
        dbSizeData["value"]["last"] = dbSize_.getLast();
        dbSize_.reset();
    }
    dbSizeData["filename"] = dbQueue_->getDbFilename();
    telemetry_->reportEvent("sqliteDatabaseSize", jsonToString(dbSizeData), true);
}

void ClickdaemonConsumer::setEnabled(bool value) {
    enabled_ = value;
    metricaRouter_->setEnabled(value);

    YIO_LOG_INFO("Clickdaemon consumer is now " << (enabled_ ? "enabled" : "disabled"));
}
