#include "clickdaemon_sender.h"

#include <yandex_io/libs/metrica/base/utils.h>

#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/counters/json_utils.h>

#include <yandex_io/libs/protobuf_utils/debug.h>

using namespace quasar;

ClickdaemonSender::ClickdaemonSender(
    std::string endpointUri, size_t batchSize,
    std::shared_ptr<MetricaBlockingQueue> immediateQueue,
    std::shared_ptr<EventsDatabase> dbQueue,
    ClickdaemonMetadata metadata,
    std::shared_ptr<YandexIO::IDevice> device)
    : endpointUri_(endpointUri)
    , isStopped_(false)
    , immediateQueue_(std::move(immediateQueue))
    , dbQueue_(std::move(dbQueue))
    , deviceInfo_{
          .deviceId = device->deviceId(),
          .platform = device->configuration()->getDeviceType(),
          .softwareVersion = device->softwareVersion(),
          .uuid = metadata.UUID}
    , realSender_(device, [this](int retryNum) {
        return delayTimingsPolicy_.calcHttpClientRetriesDelay(retryNum);
    })
    , immediateSend_(&ClickdaemonSender::immediateSendLoop, this)
    , eventsBatchSize_(batchSize)
{
    const std::chrono::milliseconds immediateQueueDelay{5000};

    delayTimingsPolicy_ = BackoffRetriesWithRandomPolicy(getCrc32(device->deviceId()) + getNowTimestampMs());
    delayTimingsPolicy_.initCheckPeriod(immediateQueueDelay, immediateQueueDelay);
}

ClickdaemonSender::~ClickdaemonSender() {
    isStopped_ = true;
    cond_.notify_all();
    immediateQueue_->push(StopQueueEvent());
    immediateSend_.join();
}

void ClickdaemonSender::setEndpointUri(const std::string& value) {
    std::lock_guard<std::mutex> lock(endpointUriMutex_);
    endpointUri_ = value;
}

std::string ClickdaemonSender::getEndpointUri() const {
    std::lock_guard<std::mutex> lock(endpointUriMutex_);
    return endpointUri_;
}

void ClickdaemonSender::setEventsBatchSize(size_t value) {
    eventsBatchSize_ = value;
}

size_t ClickdaemonSender::makePersistent() {
    auto importantEvents = (std::lock_guard(eventListsMutex_), importantEvents_);
    if (importantEvents && !importantEvents->empty()) {
        auto res = dbQueue_->makePersistent(
            [&](auto /*id*/, const auto& event, bool /*saved*/) {
                return event.has_new_environment() || importantEvents->count(event.new_event().name());
            }, maxEventId_);
        if (res.eventMaxId) {
            maxEventId_ = std::max(res.eventMaxId, maxEventId_);
        }
        if (res.eventCount) {
            YIO_LOG_INFO("To prevent the loss of " << res.eventCount << " events (" << res.eventBytes << " bytes) these saved into persistent database");
        }
        return res.eventCount;
    }
    return 0;
}

void ClickdaemonSender::immediateSendLoop() {
    while (!isStopped_) {
        MetricaQueueElement element;
        immediateQueue_->pop(element);
        YIO_LOG_INFO("Immediate element popped");
        DatabaseMetricaEvent event;

        try {
            event = std::get<DatabaseMetricaEvent>(element);
            YIO_LOG_INFO("immediate pop " << shortUtf8DebugString(event));
        } catch (const std::bad_variant_access&) {
            YIO_LOG_INFO("Got stop event from immediate queue");
            break;
        }

        std::map<std::string, std::string> environmentValues;
        for (const auto& item : event.new_environment().environment_values()) {
            environmentValues[item.first] = item.second;
        }

        bool wasSent = false;
        while (!isStopped_ && !(wasSent = sendEvents({event}, environmentValues))) {
            YIO_LOG_WARN("Unable to send immediate event: " << event.new_event().name());

            std::unique_lock<std::mutex> lock(mutex_);
            cond_.wait_for(lock, delayTimingsPolicy_.getDelayBetweenCalls(),
                           [this]() { return isStopped_; });
        }

        if (wasSent) {
            dailyStats_.increment(Counters::IMMEDIATE);
        } else {
            YIO_LOG_INFO("Wasnt sent!");
        }
    }

    YIO_LOG_INFO("Immediate send loop is stopped");
}

size_t ClickdaemonSender::flushEvents() {
    std::vector<DatabaseMetricaEvent> eventsToSend;

    std::vector<uint64_t> idsToSkip;
    std::vector<uint64_t> idsToDelete;

    bool hasEnvironment = false;
    uint64_t environmentDbId = 0;
    std::map<std::string, std::string> environmentValues;

    std::unique_ptr<EventsDatabase::Event> dbEvent;
    uint64_t maxId = 0;
    while (dbEvent = dbQueue_->getEarliestEvent(idsToSkip)) {
        auto& event = dbEvent->databaseMetricaEvent;

        maxId = std::max(maxId, dbEvent->id);
        if (event.has_new_environment()) {
            if (hasEnvironment) {
                // defer delete previous environment from db
                idsToDelete.push_back(environmentDbId);
            }

            if (eventsToSend.empty()) {
                // previous environment got no events
                // just replace it and continue to fetch events

                hasEnvironment = true;
                environmentDbId = dbEvent->id;
                environmentValues.clear();
                for (const auto& item : dbEvent->databaseMetricaEvent.new_environment().environment_values()) {
                    environmentValues[item.first] = item.second;
                }
                idsToSkip.push_back(dbEvent->id);
                continue;
            } else {
                break;
            }
        }

        eventsToSend.push_back(event);
        idsToDelete.push_back(dbEvent->id);
        idsToSkip.push_back(dbEvent->id);

        if (eventsToSend.size() >= eventsBatchSize_) {
            YIO_LOG_DEBUG("Maximum message batch size reached: size=" << eventsToSend.size());
            break;
        }
    }

    if (eventsToSend.empty()) {
        YIO_LOG_DEBUG("Got no events from database, don't send anything");
        return 0;
    }

    const auto numberOfEventsToSend = eventsToSend.size();
    if (sendEvents(std::move(eventsToSend), std::move(environmentValues))) {
        maxEventId_ = maxId;
        dbQueue_->deleteEvents(idsToDelete);
        dailyStats_.increment(StatsCounter(Counters::FROMDB, numberOfEventsToSend));
        return numberOfEventsToSend;
    }

    makePersistent();
    YIO_LOG_WARN("Unable to send events from database");
    return 0;
}

bool ClickdaemonSender::sendEvents(std::vector<quasar::proto::DatabaseMetricaEvent> events, std::map<std::string, std::string> environmentVariables) {
    environmentVariables = filterEnvironment(environmentVariables);
    return realSender_.sendEvents(getEndpointUri(), std::move(events), std::move(environmentVariables), deviceInfo_);
}

std::map<std::string, std::string> ClickdaemonSender::filterEnvironment(const std::map<std::string, std::string>& environmentVariables) {
    std::map<std::string, std::string> res;
    std::lock_guard lock(eventListsMutex_);

    for (const auto& env : environmentVariables) {
        if (envKeysBlacklist_.empty() || envKeysBlacklist_.find(env.first) == envKeysBlacklist_.end()) {
            res.emplace(env);
        }
    }

    return res;
}

void ClickdaemonSender::setEnvKeysBlacklist(std::unordered_set<std::string> envKeysBlacklist) {
    std::lock_guard lock(eventListsMutex_);
    envKeysBlacklist_ = std::move(envKeysBlacklist);
}

void ClickdaemonSender::setEventImportantList(std::shared_ptr<const std::unordered_set<std::string>> importantEvents) {
    std::lock_guard lock(eventListsMutex_);
    importantEvents_ = std::move(importantEvents);
}

Json::Value ClickdaemonSender::getStatsPayload() {
    return EnumCountersToJson(dailyStats_.getUpdatedCounters(), {"immediate", "db"});
}
