#include "clickdaemon_sender_base.h"

#include <yandex_io/libs/metrica/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

using namespace quasar;

namespace {
    /* Should be in sync with https://a.yandex-team.ru/arc/trunk/arcadia/infra/proto_logger/api/api.proto?rev=r7740250#L15 */
    enum ConnectionType: int {
        CONNECTION_CELL = 0,
        CONNECTION_WIFI = 1,
        CONNECTION_UNDEFINED = 2,
        CONNECTION_BLUETOOTH = 3,
        CONNECTION_ETHERNET = 4
    };

    ConnectionType toClickdaemonConnectionType(quasar::proto::ConnectionType connectionType) {
        switch (connectionType) {
            case quasar::proto::CONNECTION_TYPE_WIFI:
                return ConnectionType::CONNECTION_WIFI;
            case quasar::proto::CONNECTION_TYPE_ETHERNET:
                return ConnectionType::CONNECTION_ETHERNET;
            case quasar::proto::CONNECTION_TYPE_UNKNOWN:
                return ConnectionType::CONNECTION_UNDEFINED;
        }
        return ConnectionType::CONNECTION_UNDEFINED;
    }

    Json::Value environmentToJson(const std::map<std::string, std::string>& environmentVariables) {
        Json::Value res(Json::objectValue);
        for (const auto& env : environmentVariables) {
            res[env.first] = env.second;
        }
        return res;
    }

    Json::Value environmentToJson(const proto::DatabaseMetricaEvent::NewEnvironment& environmentVariables) {
        Json::Value res(Json::objectValue);
        for (const auto& env : environmentVariables.environment_values()) {
            res[env.first] = env.second;
        }
        return res;
    }

    Json::Value eventToJson(const quasar::proto::DatabaseMetricaEvent::NewEvent& event, const Json::Value& commonData) {
        Json::Value res = commonData;

        res["MetricName"] = event.name();
        res["MetricValue"] = event.value();
        res["Timestamp"] = event.timestamp();
        res["SessionId"] = event.session_id();
        res["EventNumber"] = event.serial_number();
        res["SetraceReqId"] = "";
        res["ConnectionType"] = toClickdaemonConnectionType(event.connection_type());
        if (event.has_value()) {
            if (const auto eventValue = tryParseJson(event.value()); eventValue) {
                if (eventValue->isMember("setraceMessageId")) {
                    res["SetraceReqId"] = (*eventValue)["setraceMessageId"];
                } else if (eventValue->isMember("setraceRequestId")) {
                    res["SetraceClientReqId"] = (*eventValue)["setraceRequestId"];
                }
            }
        }

        return res;
    }

    std::tuple<std::string, std::string> getQuasmodromGroup(auto environmentVariables) {
        const auto groupIter = environmentVariables.find("quasmodrom_group");
        const auto subgroupIter = environmentVariables.find("quasmodrom_subgroup");

        const auto quasmodromGroup = (groupIter != environmentVariables.end()) ? groupIter->second : "unknown";
        const auto quasmodromSubgroup = (subgroupIter != environmentVariables.end()) ? subgroupIter->second : "";
        return {quasmodromGroup, quasmodromSubgroup};
    }

    std::tuple<std::string, std::string> getQuasmodromGroup(const proto::DatabaseMetricaEvent::NewEnvironment& env) {
        return getQuasmodromGroup(env.environment_values());
    }

    Json::Value makeCommonData(Json::StreamWriterBuilder& wbuilder, auto environmentVariables, const ClickdaemonSenderBase::DeviceInfo& deviceInfo) {
        const auto environment = Json::writeString(wbuilder, environmentToJson(environmentVariables));

        const auto [quasmodromGroup, quasmodromSubgroup] = getQuasmodromGroup(environmentVariables);

        Json::Value res(Json::objectValue);

        res["DeviceId"] = deviceInfo.deviceId;
        res["Platform"] = deviceInfo.platform;
        res["QuasmodromGroup"] = quasmodromGroup;
        res["QuasmodromSubgroup"] = quasmodromSubgroup;
        res["SoftwareVersion"] = deviceInfo.softwareVersion;
        res["Environment"] = environment;
        res["UUID"] = deviceInfo.uuid;
        return res;
    }

    std::string makeCompressedPayload(std::vector<quasar::proto::DatabaseMetricaEvent> events, std::map<std::string, std::string> environmentVariables, const ClickdaemonSenderBase::DeviceInfo& deviceInfo) {
        Json::Value eventBatch(Json::arrayValue);

        Json::StreamWriterBuilder wbuilder;
        wbuilder["indentation"] = "";
        Json::Value commonData = makeCommonData(wbuilder, std::move(environmentVariables), deviceInfo);

        for (auto& event : events) {
            eventBatch.append(eventToJson(event.new_event(), commonData));
        }

        const auto request = Json::writeString(wbuilder, eventBatch);
        return compress_string(request);
    }

    std::string makeCompressedPayload(const std::vector<ClickdaemonSenderBase::EventsWithCommonEnv>& events, const ClickdaemonSenderBase::DeviceInfo& deviceInfo) {
        Json::Value eventBatch(Json::arrayValue);

        Json::StreamWriterBuilder wbuilder;
        wbuilder["indentation"] = "";

        for (auto subBatch : events) {
            Json::Value commonData = makeCommonData(wbuilder, subBatch.env, deviceInfo);
            for (auto event : subBatch.events) {
                eventBatch.append(eventToJson(event, commonData));
            }
        }

        const auto request = Json::writeString(wbuilder, eventBatch);
        return compress_string(request);
    }

} // namespace

ClickdaemonSenderBase::ClickdaemonSenderBase(std::shared_ptr<YandexIO::IDevice> device, HttpClient::CalcRetryDelayFunction delayFunc, bool keepAlive)
    : httpClient_("clickdaemon", std::move(device))
{
    httpClient_.setTimeout(std::chrono::seconds{5});
    httpClient_.setRetriesCount(3);
    // disable http_request metrics to avoid infinite loop when metrica sender
    // triggers new metrica to send
    httpClient_.disableHttpRequestMetrics();
    if (delayFunc) {
        httpClient_.setCalcRetryDelayFunction(std::move(delayFunc));
    }
    httpClient_.setReuse(keepAlive);
}

ClickdaemonSenderBase::ClickdaemonSenderBase(std::shared_ptr<YandexIO::IDevice> device, HttpClient::CalcRetryDelayFunction delayFunc)
    : ClickdaemonSenderBase(device, delayFunc, false)
{
}

ClickdaemonSenderBase::~ClickdaemonSenderBase() {
    httpClient_.cancelRetries();
}

bool ClickdaemonSenderBase::sendEvents(const std::string& endpointUri, const std::vector<EventsWithCommonEnv>& events, const DeviceInfo& deviceInfo) {
    const auto compressedRequest = makeCompressedPayload(events, deviceInfo);

    return realSend(endpointUri, compressedRequest);
}

bool ClickdaemonSenderBase::sendEvents(const std::string& endpointUri, std::vector<quasar::proto::DatabaseMetricaEvent> events, std::map<std::string, std::string> environmentVariables, const DeviceInfo& deviceInfo) {
    const auto compressedRequest = makeCompressedPayload(std::move(events), std::move(environmentVariables), deviceInfo);

    return realSend(endpointUri, compressedRequest);
}

bool ClickdaemonSenderBase::realSend(const std::string& endpointUri, const std::string& compressedRequest) {
    HttpClient::Headers headers{
        {"Expect", ""}, // Disable Expect: 100-continue behaviour
        {"Content-Encoding", "gzip"}};

    try {
        auto response = httpClient_.post("write-json-batch", endpointUri, compressedRequest, headers);

        if (response.responseCode == 200) {
            return true;
        } else {
            YIO_LOG_WARN("Got non-200 response from " << endpointUri
                                                      << ": [" << response.responseCode << "] " << response.body);
            return false;
        }
    } catch (const std::runtime_error& e) {
        YIO_LOG_WARN("Error while sending events to " << endpointUri << ": " << e.what());
        return false;
    }
}
