#pragma once

#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/libs/threading/i_callback_queue.h>

namespace YandexIO {
    namespace EventsDB {
        /* Filter can work as 'whitelist' that accept all events which names in 'eventNames'.
           When 'isWhiteList' is true. All names absent in list will be dropped by filter.
           Or it can work as 'blacklist' that block all events which names in 'eventNames'.
           When 'isWhiteList' is false. All names absent in list will be passed through the filter
         */
        struct EventsFilter {
            std::set<std::string> eventNames;
            bool isWhiteList;
            std::set<std::string> envBlackList; // this env names will no be passed to sink
        };

        enum Priority {
            LOWEST,
            LOW,
            DEFAULT,
            HIGH,
            HIGHEST,
        };

        struct Config {
            // if unset all events have same default priority
            std::function<Priority(const std::string_view&)> priority;
        };

        /*!
         * Interface to controll events coming to sink
         */
        class ISourceControl {
        public:
            virtual void readyForNext() = 0;
            virtual void releaseBeforeLast() = 0;
            virtual void releaseIncludingLast() = 0;
            virtual ~ISourceControl() = default;
        };

        using Event = quasar::proto::DatabaseMetricaEvent::NewEvent;
        using Environment = quasar::proto::DatabaseMetricaEvent::NewEnvironment;

        class Sink {
        public:
            virtual void handleDbEvent(ISourceControl& ctrl, Event event, Environment env) = 0;
            virtual ~Sink() = default;
        };
    } // namespace EventsDB

    class ITelemetryEventsDB {
    public:
        using ISourceControl = EventsDB::ISourceControl;
        using EventsFilter = EventsDB::EventsFilter;

        using Event = EventsDB::Event;
        using Environment = EventsDB::Environment;

        virtual void updateEnvironmentVar(const std::string& name, const std::string& value) = 0;
        virtual void removeEnvironmentVar(const std::string& name) = 0;
        virtual void pushEvent(const Event& event) = 0;
        virtual std::shared_ptr<ISourceControl> registerSink(unsigned idx, std::shared_ptr<EventsDB::Sink> sink, std::shared_ptr<quasar::ICallbackQueue> queue, const EventsFilter& filter) = 0;
        virtual void updateFilter(unsigned sinkIdx, const EventsFilter& newFilter) = 0;
        virtual void setConfig(const EventsDB::Config& config) = 0;

        virtual ~ITelemetryEventsDB() = default;
    };
} // namespace YandexIO
