#include <yandex_io/libs/metrica/db/i_events_db.h>
#include <yandex_io/libs/metrica/db/lmdb_events.h>

#include <yandex_io/libs/lmdb/lmdb_dbi.h>
#include <yandex_io/libs/logging/logging.h>
#include <contrib/libs/lmdbxx/lmdb++.h>

#include <google/protobuf/io/zero_copy_stream_impl_lite.h>
#include <google/protobuf/map.h>
#include <array>
#include <list>
#include <future>

#include "lmdb_events.h"

using namespace YandexIO;
using namespace quasar;
using namespace YandexIO::EventsDB;

namespace {
    // some helpers
    size_t nearestPages(size_t size) {
        const static size_t pageSize = 4096;
        auto pages = size / pageSize;
        auto result = pages * pageSize;
        if (result < size) {
            result += pageSize;
        }
        return result + pageSize;
    }

    size_t getMapSize(lmdb::env& env) {
        MDB_envinfo info;
        lmdb::env_info(env.handle(), &info);
        return info.me_mapsize;
    }

    lmdb::env initEnv(const std::string& dbDir, std::uint64_t dbSize) {
        auto env = lmdb::env::create();
        env.set_mapsize(dbSize);
        env.set_max_dbs(3); // events, envs, cfg
        env.open(dbDir.c_str(), MDB_CREATE);
        return env;
    }

    std::uint64_t getLastId(lmdb::env& env, auto& dbi) {
        auto txn = lmdb::txn::begin(env, nullptr, MDB_RDONLY);
        {
            auto cursor = dbi.openCursor(txn);
            if (auto key = cursor.last()) {
                return key->idx;
            }
        }
        return 0;
    }

    bool filterAccept(const EventsFilter& filter, const std::string& name) {
        return filter.eventNames.contains(name) ? filter.isWhiteList : !filter.isWhiteList;
    }

    std::string serialize(const auto& event) {
        TString tmpStr;
        Y_PROTOBUF_SUPPRESS_NODISCARD event.SerializeToString(&tmpStr);
        return tmpStr;
    }

    bool deserialize(std::string_view src, auto& dst) {
        google::protobuf::io::ArrayInputStream stream(src.data(), src.size());
        return dst.ParseFromZeroCopyStream(&stream);
    }

    void filterEnvironment(ITelemetryEventsDB::Environment& env, const std::set<std::string>& blackList) {
        auto vals = env.mutable_environment_values();
        for (const auto& name : blackList) {
            vals->erase(name);
        }
    }

    std::string getEventName(const ITelemetryEventsDB::Event& event) {
        if (event.has_name()) {
            return event.name();
        }
        return std::string();
    }

    constexpr const char* DB_NAME_EVENTS{"events"};
    constexpr const char* DB_NAME_ENV{"env"};
    constexpr const char* DB_NAME_CFG{"cfg"};
} // namespace

LMDBEventsDB::SinkSourceControl::SinkSourceControl(LMDBEventsDB& db,
                                                   std::shared_ptr<EventsDB::Sink> sink,
                                                   std::shared_ptr<ICallbackQueue> queue,
                                                   std::uint64_t lastKey,
                                                   const EventsFilter& filter,
                                                   unsigned idx)
    : idx_(idx)
    , db_(db)
    , sink_(std::move(sink))
    , queue_(std::move(queue))
    , lastKey_(lastKey)
    , filter_(filter)
{
}

Environment LMDBEventsDB::SinkSourceControl::getEnv(std::uint64_t envId) {
    if (loadedEnvId_ != envId) {
        auto txn = lmdb::txn::begin(db_.env_, nullptr, MDB_RDONLY);
        {
            auto cursor = db_.envDbi_.openCursor(txn);
            if (auto keyVal = cursor.toKeyVal(envId)) {
                loadedEnvId_ = envId;
                loadedEnv_ = Environment();
                if (keyVal->key == envId) {
                    if (deserialize(keyVal->value, loadedEnv_)) {
                        filterEnvironment(loadedEnv_, filter_.envBlackList);
                    }
                }
            }
        }
    }
    return loadedEnv_;
}

void LMDBEventsDB::SinkSourceControl::bypass(const Event& event, const Environment& env) {
    eof_ = false;
    queue_->add([this, event, env]() {
        auto envCopy = env;
        filterEnvironment(envCopy, filter_.envBlackList);
        sink_->handleDbEvent(*this, event, envCopy);
    });
}

void LMDBEventsDB::SinkSourceControl::readyForNext() {
    queue_->add([this]() {
        auto txn = lmdb::txn::begin(db_.env_, nullptr, MDB_RDONLY);
        {
            DBKey pos{
                .idx = lastKey_ + 1,
            };
            auto cursor = db_.eventDbi_.openCursor(txn);
            YIO_LOG_TRACE("ask " << pos.idx);
            if (auto keyVal = cursor.toKeyVal(pos)) {
                auto [dbKey, val] = *keyVal;
                lastKey_ = dbKey.idx;
                if (!(dbKey.mask & (1 << idx_))) {
                    queue_->add([this]() {
                        readyForNext();
                    });
                    return; // skip this event cos it filtered by our filter
                }
                YIO_LOG_TRACE("Got event " << lastKey_ << " mask=" << dbKey.mask << " vs " << idx_);
                Event event;
                {
                    if (deserialize(val, event)) {
                        queue_->add([this, envId = dbKey.env, event = std::move(event)]() {
                            sink_->handleDbEvent(*this, event, getEnv(envId));
                        });
                    }
                }
            } else {
                eof_ = true;
            }
        }
    });
}

void LMDBEventsDB::SinkSourceControl::releaseBeforeLast() {
    readyToRelease_ = lastKey_ - 1;
    db_.pushReleases();
}

void LMDBEventsDB::SinkSourceControl::releaseIncludingLast() {
    readyToRelease_ = lastKey_;
    db_.pushReleases();
}

void LMDBEventsDB::SinkSourceControl::updateFilter(const EventsFilter& newFilter) {
    std::promise<void> promise;
    queue_->add([this, &newFilter, &promise] { // to safely use envfilter in queue_
        filter_ = newFilter;
        promise.set_value();
    });
    promise.get_future().get();
}

void LMDBEventsDB::SinkSourceControl::continueAfterEof() {
    if (!eof_) {
        return;
    }
    queue_->add([this]() {
        if (eof_) {
            eof_ = false;
            readyForNext();
        }
    });
}

LMDBEventsDB::LMDBEventsDB(const std::string& dbDir, std::uint64_t dbSize, std::shared_ptr<quasar::ICallbackQueue> queue)
    : dbDir_(dbDir)
    , dbSize_(dbSize)
    , env_(initEnv(dbDir, nearestPages(dbSize * 1.2 + 100000)))
    , eventDbi_(env_, DB_NAME_EVENTS)
    , envDbi_(env_, DB_NAME_ENV)
    , cfgDbi_(env_, DB_NAME_CFG)
    , stats_(calculateDbStatistics())
    , nextId_(getLastId(env_, eventDbi_) + 1)
    , dbQueue_(std::move(queue))
    , eofsFlusher_(dbQueue_, [this]() {
        flushEOFed();
    }, true)
    , releaser_(dbQueue_, [this]() {
        release();
    }, false)
{
}

void LMDBEventsDB::updateEnvironmentVar(const std::string& name, const std::string& value) {
    dbQueue_->add([this, name, value]() {
        auto vals = currentEnv_.mutable_environment_values();
        auto [iter, inserted] = vals->insert(google::protobuf::MapPair<TString, TString>{TString(name), TString(value)});
        if (!inserted) {
            if (iter->second == value) {
                return; // value isnt changed
            }
            iter->second = value;
        }
        currentEnvStored_ = false;
    });
}

void LMDBEventsDB::removeEnvironmentVar(const std::string& name) {
    dbQueue_->add([this, name]() {
        auto vals = currentEnv_.mutable_environment_values();
        if (vals->erase(name)) {
            currentEnvStored_ = false;
        }
    });
}

LMDBEventsDB::Counters LMDBEventsDB::calculateDbStatistics() {
    Counters result;
    std::array<unsigned, 5> prioCounts{};
    auto txn = lmdb::txn::begin(env_, nullptr, MDB_RDONLY);
    eventDbi_.forEach(txn, [this, &prioCounts, &result](const DBKey& key, const std::string_view& value) {
        prioCounts[key.prio] += 1;
        result.addEvent(value.size());
        if (!firstKey_) {
            firstKey_ = key.idx;
        }
    });
    YIO_LOG_DEBUG("First events key " << firstKey_);

    unsigned envsCount = 0;
    std::uint64_t envsSize = 0;
    envDbi_.forEach(txn, [this, &envsCount, &envsSize](const std::uint64_t& key, const std::string_view& value) {
        envsSize += value.size();
        ++envsCount;
        nextEnvId_ = key + 1;
    });

    cfgDbi_.forEach(txn, [](const std::uint64_t& key, const std::string_view& value) {
        YIO_LOG_DEBUG("Last key for sink with idx = " << key << " is " << lmdb::from_sv<std::uint64_t>(value));
    });
    txn.abort();

    YIO_LOG_INFO("Envs in db: " << envsCount << " of size " << envsSize << " next id " << nextEnvId_);

    for (unsigned i = 0; i < prioCounts.size(); ++i) {
        YIO_LOG_INFO("Prio " << i << " - " << prioCounts[i] << " events");
    }
    YIO_LOG_DEBUG("Reserved space " << reservedSpace_);
    return result;
}

void LMDBEventsDB::release() {
    Y_ENSURE_THREAD(dbQueue_);

    std::vector<std::tuple<std::shared_ptr<SinkSourceControl>, std::uint64_t>> releaseOrder;

    for (auto& sink : sinks_) {
        releaseOrder.emplace_back(sink, sink->readyToRelease_.load());
    }

    // reverse order cos we go from end of db to begin
    std::sort(releaseOrder.begin(), releaseOrder.end(), [](auto& a, auto& b) {
        return std::get<1>(a) < std::get<1>(b);
    });

    if (releaseOrder.empty() || firstKey_ > std::get<1>(releaseOrder.back())) {
        YIO_LOG_DEBUG("Releasing skipped");
        return;
    }

    std::vector<std::uint64_t> keysToRemove;
    std::uint32_t sinksNotPassedYet = 0;
    auto orderIter = releaseOrder.begin();
    auto nextRange = std::get<1>(*orderIter);
    YIO_LOG_DEBUG("Start collecting from " << firstKey_ << " to " << std::get<1>(releaseOrder.back()));
    {
        auto txn = lmdb::txn::begin(env_, nullptr, MDB_RDONLY);
        eventDbi_.forEach(txn, [&](const DBKey& key, const std::string_view& /*value*/) -> bool {
            if (key.idx > nextRange) {
                return false;
            }
            if (!(sinksNotPassedYet & key.mask)) {
                keysToRemove.push_back(key.idx);
            }
            if (key.idx == nextRange) { // advance over releaseOrder for new nextRange value
                do {
                    sinksNotPassedYet |= 1 << std::get<0>(*orderIter)->idx_;
                    if (++orderIter == releaseOrder.end()) {
                        break;
                    }
                    auto rangeCadidate = std::get<1>(*orderIter);
                    if (rangeCadidate != nextRange) {
                        nextRange = rangeCadidate;
                        break;
                    }
                } while (true);
                if (orderIter == releaseOrder.end()) {
                    return false;
                }
            }
            return true;
        });
        txn.abort();
    }
    if (!keysToRemove.empty()) {
        realRemove(keysToRemove);
    }
    auto txn = lmdb::txn::begin(env_);
    for (auto [sink, releasedKey] : releaseOrder) {
        {
            auto cursor = cfgDbi_.openCursor(txn);
            auto key = cursor.toKey(sink->idx_);
            if (key && *key == sink->idx_) { // replace
                cursor.replace(*key, lmdb::to_sv(releasedKey));
                continue;
            };
        }
        cfgDbi_.put(txn, sink->idx_, lmdb::to_sv(releasedKey));
    }
    txn.commit();
    YIO_LOG_DEBUG("Tail cleanup completed");
}

void LMDBEventsDB::pushReleases() {
    releaser_.call();
}

void LMDBEventsDB::flushEOFed() {
    for (auto& sink : sinks_) {
        sink->continueAfterEof();
    }
}

std::uint8_t LMDBEventsDB::getPriority(const Event& event) const {
    if (config_.priority) {
        return config_.priority(event.name());
    }
    return Priority::DEFAULT;
}

void LMDBEventsDB::realRemove(const std::vector<std::uint64_t>& keys) {
    while (true) {
        try {
            eventDbi_.remove(keys, env_, [this](const std::string_view& value) {
                stats_.removeEvent(value.size());
            });
            break;
        } catch (std::runtime_error& err) {
            YIO_LOG_DEBUG("err during remove " << err.what());
        }
        incrementDbSize(4096); // page
    }
    std::uint64_t notUsedEnvs = nextEnvId_ - 1;
    {
        auto txn = lmdb::txn::begin(env_);
        {
            auto cursor = eventDbi_.openCursor(txn);
            if (auto key = cursor.first()) {
                firstKey_ = key->idx;
                notUsedEnvs = key->env;
            } else {
                firstKey_ = nextId_;
            }
            YIO_LOG_DEBUG("New first key " << firstKey_ << ", notUsedEnvs = " << notUsedEnvs);
        }
    }
    freeEnvOlder(notUsedEnvs);
}

// return true if some data was removed from db
bool LMDBEventsDB::priorityCleanup(Priority below, unsigned neededSize) {
    Counters freed;
    std::vector<std::uint64_t> cleanupKeys;
    {
        auto txn = lmdb::txn::begin(env_);
        Priority nextPrio = Priority::LOWEST;
        do {
            eventDbi_.forEachReverse(txn, [&](const DBKey& key, const std::string_view& value) {
                if (key.prio == nextPrio) {
                    cleanupKeys.push_back(key.idx);
                    freed.addEvent(value.size());
                    if (freed.valuesSize > neededSize) {
                        return false;
                    }
                }
                return true;
            });
            nextPrio = static_cast<Priority>(static_cast<int>(nextPrio) + 1);
        } while (nextPrio < below && freed.valuesSize < neededSize);
        txn.abort();
    }
    if (freed.valuesSize >= neededSize) {
        realRemove(cleanupKeys);
    } else {
        YIO_LOG_DEBUG("Priority " << (int)below << " cleanup aborted cos not enough size will be freed " << freed.valuesSize << " but needed " << neededSize);
        return false;
    }
    return true;
}

void LMDBEventsDB::incrementDbSize(unsigned increment) {
    const auto newDbSize = nearestPages(getMapSize(env_) + increment * 2);
    YIO_LOG_WARN("Incrementing mapsize to " << newDbSize << " but decrementing dbSize_ to reached value " << stats_.valuesSize);
    env_.close();
    env_ = initEnv(dbDir_, newDbSize);
    eventDbi_ = {env_, DB_NAME_EVENTS};
    envDbi_ = {env_, DB_NAME_ENV};
    dbSize_ = stats_.valuesSize;
}

std::uint64_t LMDBEventsDB::pagedDbSize(lmdb::txn& txn) {
    MDB_stat info = eventDbi_.stat(txn);
    return info.ms_psize * (info.ms_branch_pages + info.ms_leaf_pages + info.ms_overflow_pages);
}

std::uint32_t LMDBEventsDB::filteredMask(const Event& event) const {
    std::uint32_t result = 0;
    for (auto sink : sinks_) {
        if (filterAccept(sink->filter_, event.name())) {
            result |= 1 << sink->idx_;
        }
    }
    return result;
}

void LMDBEventsDB::pushToEOFed(const Event& event, std::uint32_t filterMask) {
    for (auto sink : sinks_) {
        if (sink->eof_ && (filterMask & (1 << sink->idx_))) { // eof and filter accept
            sink->bypass(event, currentEnv_);
        }
    }
}

std::uint64_t LMDBEventsDB::estimateEventStorageSize(std::uint64_t valueSize, std::string& serializedEnv) {
    std::uint64_t result = sizeof(DBKey) + valueSize;
    if (!currentEnvStored_) {
        auto envCopy = currentEnv_;
        filterEnvironment(envCopy, commonEnvBlackList_);
        serializedEnv = serialize(envCopy);
        result += sizeof(std::uint64_t) + serializedEnv.size();
    }
    return result;
}

void LMDBEventsDB::pushEvent(const Event& event) {
    std::promise<void> finish;
    dbQueue_->add([this, &event, &finish]() {
        const auto filterMask = filteredMask(event);
        if (filterMask == 0) {
            YIO_LOG_DEBUG("Sinks skipped event " << event.name());
            finish.set_value();
            return;
        }

        DBKey key{
            .idx = nextId_++,
            .prio = getPriority(event),
            .mask = filterMask,
            .env = nextEnvId_ - 1,
        };
        YIO_LOG_TRACE("pushing " << EventsKeyConv::toStr(key) << stats_.events << '|' << stats_.valuesSize << "/" << dbSize_);
        auto serialized = serialize(event);
        std::string serializedEnv;
        bool inserted = false;
        const auto estimatedSize = estimateEventStorageSize(serialized.size(), serializedEnv);
        do {
            try {
                if (estimatedSize + stats_.valuesSize + reservedSpace_ < dbSize_) {
                    auto txn = lmdb::txn::begin(env_);
                    if (pagedDbSize(txn) < dbSize_ + reservedSpace_ + estimatedSize) {
                        if (!currentEnvStored_) {
                            key.env = nextEnvId_;
                            envDbi_.put(txn, nextEnvId_++, serializedEnv);
                        }
                        eventDbi_.put(txn, key, serialized);
                        txn.commit();
                        inserted = true;
                        currentEnvStored_ = true;
                        YIO_LOG_TRACE("Commited");
                        stats_.addEvent(serialized.size());
                        break;
                    }
                }
            } catch (std::runtime_error& err) {
                YIO_LOG_WARN("Failed to add into db " << err.what());
                incrementDbSize(serialized.size());
            }
            if (key.prio == Priority::LOWEST || !priorityCleanup(static_cast<Priority>(key.prio), estimatedSize)) {
                pushToEOFed(event, filterMask);
                YIO_LOG_WARN("Drop message '" << getEventName(event) << "' cos there is no room for priority " << key.prio << ". There are now lower priority left in db");
                break; // do not retry
            }
            YIO_LOG_DEBUG("previous attempt to push event was failed. retrying");
        } while (true);
        if (inserted) {
            eofsFlusher_.call();
        }
        finish.set_value();
    });
    finish.get_future().get();
}

bool LMDBEventsDB::validIdx(unsigned idx) const {
    if (idx >= 32) {
        return false;
    }
    for (auto sink : sinks_) {
        if (sink->idx_ == idx) {
            return false;
        }
    }
    return true;
}

std::uint64_t LMDBEventsDB::restoreSinkLastKey(unsigned idx) {
    auto txn = lmdb::txn::begin(env_, nullptr, MDB_RDONLY);
    {
        auto cursor = cfgDbi_.openCursor(txn);
        auto keyVal = cursor.toKeyVal(idx);
        if (keyVal && keyVal->key == idx) {
            auto result = lmdb::from_sv<std::uint64_t>(keyVal->value);
            YIO_LOG_DEBUG("restore position of sink " << idx << " as " << result << " but nextId_ is " << nextId_);
            return result >= nextId_ ? nextId_ : result;
        }
    }
    return 0;
}

void LMDBEventsDB::updateEnvBlackList() {
    Y_ENSURE_THREAD(dbQueue_);
    if (sinks_.empty()) {
        return;
    }
    auto cur = std::begin(sinks_);
    auto end = std::end(sinks_);
    std::set<std::string> common = (*cur)->filter_.envBlackList;
    while (++cur != end) {
        const auto& blackList = (*cur)->filter_.envBlackList;
        std::set<std::string> newCommon;
        std::set_intersection(std::begin(common), std::end(common),
                              std::begin(blackList), std::end(blackList),
                              std::inserter(newCommon, std::end(newCommon)));
        common.swap(newCommon);
    }
    commonEnvBlackList_.swap(common);
}

std::shared_ptr<ITelemetryEventsDB::ISourceControl>
LMDBEventsDB::registerSink(unsigned idx,
                           std::shared_ptr<EventsDB::Sink> sink,
                           std::shared_ptr<ICallbackQueue> queue,
                           const EventsFilter& filter) {
    YIO_LOG_DEBUG("Registering sink " << idx);
    std::promise<std::shared_ptr<ITelemetryEventsDB::ISourceControl>> result;
    dbQueue_->add([this, &result, sink = std::move(sink), queue = std::move(queue), &filter, idx] { // sink should be inited in its thread cos txn locks assigned to it
        if (validIdx(idx)) {
            auto handler = std::make_shared<SinkSourceControl>(*this, sink, queue, restoreSinkLastKey(idx), filter, idx);
            sinks_.push_back(handler);
            updateEnvBlackList();
            result.set_value(handler);
        } else {
            YIO_LOG_WARN("Sink not registered due to invalid idx " << idx);
            result.set_value(nullptr);
        }
    });
    return result.get_future().get();
}

void LMDBEventsDB::updateFilter(unsigned sinkIdx, const EventsFilter& newFilter) {
    std::promise<void> finish;
    dbQueue_->add([this, sinkIdx, &newFilter, &finish]() {
        for (const auto& sink : sinks_) {
            if (sink->idx_ == sinkIdx) {
                sink->updateFilter(newFilter);
                break;
            }
        }
        updateEnvBlackList();
        finish.set_value();
    });
    finish.get_future().get();
}

void LMDBEventsDB::setConfig(const EventsDB::Config& config) {
    std::promise<void> finish;
    dbQueue_->add([this, &config, &finish]() {
        config_ = config;
        finish.set_value();
    });
    finish.get_future().get();
}

// remove all envs if their ids < maxEnvId
void LMDBEventsDB::freeEnvOlder(std::uint64_t maxEnvId) {
    std::vector<std::uint64_t> keysToRemove;
    {
        auto txn = lmdb::txn::begin(env_, nullptr, MDB_RDONLY);
        {
            envDbi_.forEach(txn, [&](const std::uint64_t& idx, const std::string_view& /*value*/) {
                if (idx >= maxEnvId) {
                    return false;
                }
                keysToRemove.push_back(idx);
                return true;
            });
        }
    }
    envDbi_.remove(keysToRemove, env_, [](const std::string_view& value) {
        YIO_LOG_DEBUG("removed env of size " << value.size());
    });
}

std::shared_ptr<ITelemetryEventsDB> YandexIO::makeLmdbEventsDb(const std::string& dbDir, std::uint64_t dbSize, std::shared_ptr<quasar::ICallbackQueue> queue) {
    return std::make_shared<LMDBEventsDB>(dbDir, dbSize, std::move(queue));
}
