#pragma once

#include "throttled_callable.h"
#include <util/system/byteorder.h>

namespace YandexIO {
#pragma pack(push)
    struct DBKey {
        std::uint64_t idx;
        std::uint32_t prio{0};
        std::uint32_t mask{0};
        std::uint64_t env{0};
    };
#pragma pack(pop)

    struct EventsKeyConv {
        using KeyType = DBKey;

        static DBKey toDbKey(const DBKey& key) {
            DBKey result = key;
            result.idx = HostToInet<std::uint64_t>(key.idx);
            return result;
        }

        static DBKey indexToKey(std::uint64_t idx) {
            return DBKey{
                .idx = idx,
            };
        }

        static DBKey fromSv(const std::string_view& svKey) {
            DBKey result{
                .idx = 0,
            };
            result = lmdb::from_sv<DBKey>(svKey);
            result.idx = InetToHost<std::uint64_t>(result.idx);
            return result;
        }

        static bool equal(const DBKey& key, std::uint64_t idx) {
            return key.idx == idx;
        }

        static bool equal(const DBKey& key, const DBKey& key2) {
            return key.idx == key2.idx;
        }

        static std::string toStr(const DBKey& key) {
            // clang-format off
            return std::to_string(key.idx)
                + "[prio:" + std::to_string(key.prio)
                + ", mask:" + std::to_string(key.mask)
                + ", env:" + std::to_string(key.mask)
                + "]";
            // clang-format on
        }
    };

    struct EnvKeyConv {
        using KeyType = std::uint64_t;

        static std::uint64_t toDbKey(std::uint64_t idx) {
            return HostToInet<std::uint64_t>(idx);
        }

        static std::uint64_t indexToKey(std::uint64_t idx) {
            return idx;
        }
        static std::uint64_t fromSv(const std::string_view& svKey) {
            return InetToHost<std::uint64_t>(lmdb::from_sv<std::uint64_t>(svKey));
        }
        static bool equal(std::uint64_t key, std::uint64_t idx) {
            return key == idx;
        }

        static std::string toStr(std::uint64_t idx) {
            return std::to_string(idx);
        }
    };

    // lmdb implementation
    class LMDBEventsDB: public ITelemetryEventsDB {
    public:
        LMDBEventsDB(const std::string& dbDir, std::uint64_t dbSize, std::shared_ptr<quasar::ICallbackQueue> queue);

        void updateEnvironmentVar(const std::string& name, const std::string& value) override;
        void removeEnvironmentVar(const std::string& name) override;
        void pushEvent(const Event& /*event*/) override;
        std::shared_ptr<ISourceControl> registerSink(unsigned idx, std::shared_ptr<EventsDB::Sink> sink, std::shared_ptr<quasar::ICallbackQueue> queue, const EventsFilter& filter) override;
        void updateFilter(unsigned sinkIdx, const EventsFilter& newFilter) override;
        void setConfig(const EventsDB::Config& config) override;

    private:
        using RealKey = std::tuple<std::uint64_t, std::uint64_t>;

        struct Counters {
            std::uint64_t events{0};
            std::uint64_t valuesSize{0};

            void addEvent(std::uint64_t size) {
                ++events;
                valuesSize += size + sizeof(DBKey);
            }
            void removeEvent(std::uint64_t size) {
                --events;
                valuesSize -= size + sizeof(DBKey);
            }
            void removeEvents(const Counters& dec) {
                events -= dec.events;
                valuesSize -= dec.valuesSize;
            }
        };

        struct SinkSourceControl: public ISourceControl {
            const unsigned idx_;
            LMDBEventsDB& db_;
            std::shared_ptr<EventsDB::Sink> sink_;
            std::shared_ptr<quasar::ICallbackQueue> queue_;
            std::atomic_bool eof_{false};
            std::uint64_t lastKey_{0};
            std::atomic<std::uint64_t> readyToRelease_{0};
            std::uint64_t released_{0};
            EventsFilter filter_;
            Environment loadedEnv_;
            std::uint64_t loadedEnvId_{0};

            SinkSourceControl(LMDBEventsDB& db,
                              std::shared_ptr<EventsDB::Sink> sink,
                              std::shared_ptr<quasar::ICallbackQueue> queue,
                              std::uint64_t lastKey,
                              const EventsFilter& filter,
                              unsigned idx);

            Environment getEnv(std::uint64_t envId);
            void bypass(const Event& event, const Environment& env);

            void readyForNext() override;
            void releaseBeforeLast() override;
            void releaseIncludingLast() override;
            void updateFilter(const EventsFilter& newFilter);

            void continueAfterEof();
        };

        // ask release
        void pushReleases();
        // remove all events from db '<= key' if all released them
        std::uint64_t restoreSinkLastKey(unsigned idx);
        void release();
        void flushEOFed();
        void pushToEOFed(const Event& event, std::uint32_t filterMask);
        std::uint8_t getPriority(const Event& event) const;
        void incrementDbSize(unsigned increment);
        void realRemove(const std::vector<std::uint64_t>& keys);
        bool priorityCleanup(EventsDB::Priority below, unsigned neededSize);
        Counters calculateDbStatistics();
        std::uint64_t pagedDbSize(lmdb::txn& txn);
        std::uint32_t filteredMask(const Event& event) const;
        bool validIdx(unsigned idx) const;
        void updateEnvBlackList();

        std::uint64_t estimateEventStorageSize(std::uint64_t valueSize, std::string& serializedEnv);
        void freeEnvOlder(std::uint64_t maxEnvId);

    private:
        const std::string dbDir_;
        std::uint64_t dbSize_;
        lmdb::env env_;
        quasar::LmdbDbi<EventsKeyConv> eventDbi_;
        quasar::LmdbDbi<EnvKeyConv> envDbi_;
        quasar::LmdbDbi<EnvKeyConv> cfgDbi_;
        Environment currentEnv_;
        bool currentEnvStored_ = false;
        std::uint64_t nextEnvId_{1};

        std::uint64_t reservedSpace_{4096 * 2};
        std::uint64_t firstKey_{0};
        Counters stats_;
        std::uint64_t nextId_;

        std::set<std::string> commonEnvBlackList_;
        std::vector<std::shared_ptr<SinkSourceControl>> sinks_;
        EventsDB::Config config_;
        std::shared_ptr<quasar::ICallbackQueue> dbQueue_;
        ThrottledCallable eofsFlusher_;
        ThrottledCallable releaser_;
    };
} // namespace YandexIO
