#include "throttled_callable.h"

#include <future>

using namespace YandexIO;

void ThrottledCallable::run() {
    running_ = true;
    queue_->add([this]() {
        {
            std::scoped_lock lock(mutex_);
            shouldBeRun_ = false;
        }
        callback_();
        std::scoped_lock lock(mutex_);
        running_ = false;
        if (shouldBeRun_ && !stopped_) {
            run();
        }
    });
}

ThrottledCallable::ThrottledCallable(std::shared_ptr<quasar::ICallbackQueue> queue, std::function<void()> cb, bool doLastCall)
    : queue_(std::move(queue))
    , callback_(cb)
    , doLastCall_(doLastCall)
{
}

ThrottledCallable::~ThrottledCallable() {
    {
        std::scoped_lock lock(mutex_);
        stopped_ = true;
    }
    std::promise<void> lastCall;
    queue_->add([this, &lastCall]() {
        if (doLastCall_ && shouldBeRun_) {
            callback_();
        }
        lastCall.set_value();
    });
    lastCall.get_future().get();
}

void ThrottledCallable::call() {
    std::scoped_lock lock(mutex_);
    if (!stopped_) {
        if (running_) {
            shouldBeRun_ = true;
        } else {
            run();
        }
    }
}
