#include "disk_stats_info.h"

#include <boost/algorithm/string.hpp>

#include <fstream>
#include <stdexcept>

namespace quasar {

    DiskStatsInfo DiskStatsInfo::getDiskStatsInfo(const std::vector<std::string>& disksList, int64_t uptime)
    {
        DiskStatsInfo result;
        result.devices.reserve(disksList.size());

        std::ifstream diskstats("/proc/diskstats");
        if (!diskstats.good())
        {
            throw std::runtime_error("Error opening /proc/diskstats");
        }

        std::string line;
        while (getline(diskstats, line))
        {
            std::vector<std::string> fields;

            boost::trim_if(line, boost::is_any_of("\t "));
            boost::split(fields, line, boost::is_any_of("\t "), boost::token_compress_on);

            if (fields.size() < DiskStatsInfo::DiskStats::ST_LAST - 1)
            {
                throw std::runtime_error("Not enough fields in /proc/diskstats."
                                         " At least " +
                                         std::to_string(DiskStatsInfo::DiskStats::ST_LAST - 1) + " expected.");
            }
            if (find(disksList.begin(), disksList.end(), fields[DiskStatsInfo::DiskStats::ST_DEV_NAME]) == disksList.end())
            {
                continue;
            }

            DiskStatsInfo::DiskStats devStats;

            devStats.rdOps = atoll(fields[DiskStatsInfo::DiskStats::ST_RD_COMPLETE].c_str());
            devStats.rdSectors = atoll(fields[DiskStatsInfo::DiskStats::ST_RD_SECTORS].c_str());
            devStats.wrOps = atoll(fields[DiskStatsInfo::DiskStats::ST_WR_COMPLETE].c_str());
            devStats.wrSectors = atoll(fields[DiskStatsInfo::DiskStats::ST_WR_SECTORS].c_str());

            result.devices.emplace_back(DiskDevice{fields[DiskStatsInfo::DiskStats::ST_DEV_NAME], uptime, devStats});
        }

        return result;
    }

    int64_t DiskStatsInfo::DiskDevice::readsPerSecKB() const {
        return uptime ? (stats.rdSectors / uptime) / SECTORS_PER_KB : 0;
    }

    int64_t DiskStatsInfo::DiskDevice::writesPerSecKB() const {
        return uptime ? (stats.wrSectors / uptime) / SECTORS_PER_KB : 0;
    }

} // namespace quasar
