#include "ubi_info.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string.hpp>

#include <fstream>
#include <stdexcept>

YIO_DEFINE_LOG_MODULE("ubi_info");

namespace quasar {

    UbiInfo UbiInfo::getUbiInfo(const std::vector<std::string>& disksList, int64_t uptime)
    {
        UbiInfo result;
        result.devices.reserve(disksList.size());

        for (const auto& disk : disksList) {
            std::string filename("/sys/class/ubi/" + disk + "/stats");
            std::vector<std::string> fields;
            try {
                std::ifstream ubiStat(filename);
                if (!ubiStat.good())
                {
                    throw std::runtime_error("Can't open file");
                }

                std::string line;
                for (size_t lineNo = 0; lineNo < 2 && getline(ubiStat, line); ++lineNo)
                {
                    boost::trim_if(line, boost::is_any_of(":\t "));
                    boost::split(fields, line, boost::is_any_of("\t "), boost::token_compress_on);

                    if (fields.size() < Stats::LAST)
                    {
                        throw std::runtime_error("Not enough fields. At least " + std::to_string(Stats::LAST) + " expected.");
                    }
                    switch (lineNo) {
                        case 0:
                            if (fields[Stats::RBYTES] != "rbytes" ||
                                fields[Stats::WBYTES] != "wbytes" ||
                                fields[Stats::RCOUNT] != "rcount" ||
                                fields[Stats::WCOUNT] != "wcount")
                            {
                                throw std::runtime_error("Unexpected fields: " + join(fields, ", "));
                            }
                            break;
                        default: {
                            Stats stats;
                            stats.rbytes = boost::lexical_cast<int64_t>(fields[Stats::RBYTES]);
                            stats.wbytes = boost::lexical_cast<int64_t>(fields[Stats::WBYTES]);
                            stats.rcount = boost::lexical_cast<int64_t>(fields[Stats::RCOUNT]);
                            stats.wcount = boost::lexical_cast<int64_t>(fields[Stats::WCOUNT]);
                            result.devices.emplace_back(Device{disk, uptime, stats});
                        } break;
                    }
                }
            } catch (const std::exception& ex) {
                YIO_LOG_ERROR_EVENT("UbiInfo.GetUbiInfo", "Fail to get information from " + filename + ": " << ex.what());
            } catch (...) {
                YIO_LOG_ERROR_EVENT("UbiInfo.GetUbiInfo", "Fail to get information from " + filename + ": unexpected exception");
            }
        }

        return result;
    }

    std::vector<std::pair<std::string, int64_t>> UbiInfo::getNumericStats(const std::vector<std::string>& disksList, int64_t uptime)
    {
        UbiInfo ubiInfo = UbiInfo::getUbiInfo(disksList, uptime);
        std::vector<std::pair<std::string, int64_t>> result;
        result.reserve(4 * ubiInfo.devices.size());

        for (const auto& device : ubiInfo.devices)
        {
            result.emplace_back(std::make_pair(device.name + "-readPerSecKB", device.readsPerSecKB()));
            result.emplace_back(std::make_pair(device.name + "-writePerSecKB", device.writesPerSecKB()));
            result.emplace_back(std::make_pair(device.name + "-readBytes", device.stats.rbytes));
            result.emplace_back(std::make_pair(device.name + "-writeBytes", device.stats.wbytes));
        }
        return result;
    }

    int64_t UbiInfo::Device::readsPerSecKB() const {
        return uptime ? (stats.rbytes / uptime) / 1024 : 0;
    }

    int64_t UbiInfo::Device::writesPerSecKB() const {
        return uptime ? (stats.wbytes / uptime) / 1024 : 0;
    }

} // namespace quasar
