#pragma once

#include "numeric_stat.h"

#include <json/json.h>

#include <string_view>

namespace quasar {

    // A metric consumer accepts metric values from a provider and forwards them to
    // the underlying data model
    //
    // For the desired data model and naming, please try to follow Solomon metric data model:
    //
    //  https://docs.yandex-team.ru/solomon/concepts/data-model
    //
    // Even if our data model does not map 1:1 to Solomon's one, similar features
    // should be compatible.
    class IMetricConsumer {
    public:
        virtual ~IMetricConsumer();

        // All data ingestion methods accept name as a string_view to support
        // compile-time static strings without extra allocations.

        // TODO: Add support for metric labels, as supported by
        // Solomon/Prometheus/OpenTSDB data models:
        //
        // struct LabelSet { ... };
        // virtual void addFoo(std::string_view name, T value, const LabelSet* optLabels) = 0;

        // A snapshot of a 'real' numeric value, e.g. CPU temperature
        virtual void addDGauge(std::string_view name, double value) = 0;

        // A snapshot of an 'integer' numeric value, e.g. disk free space
        virtual void addIGauge(std::string_view name, int64_t value) = 0;

        // A statistic summary of a 'real' numeric value, e.g. CPU temperature
        virtual void addDGaugeStat(std::string_view name, const NumericStat<double>& value) = 0;

        // A statistic summary of an 'integer' numeric value, e.g. disk free space
        virtual void addIGaugeStat(std::string_view name, const NumericStat<int64_t>& value) = 0;

        // An arbitrary JSON payload
        virtual void addJson(std::string_view name, Json::Value value) = 0;
    };

} // namespace quasar
