#include "quasar_minidump.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/telemetry/telemetry.h>

#include <contrib/libs/breakpad/src/client/linux/handler/exception_handler.h>

#include <json/json.h>

#include <util/system/yassert.h>

#include <atomic>
#include <fstream>
#include <memory>
#include <cstring>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

YIO_DEFINE_LOG_MODULE("minidump");

using namespace quasar;
using namespace google_breakpad;

QuasarMinidump& QuasarMinidump::getInstance()
{
    static QuasarMinidump instance;

    return instance;
}

bool QuasarMinidump::dumpCallback(const MinidumpDescriptor& descriptor,
                                  void* context,
                                  bool succeeded)
{
    /* NOTE: Avoid any allocations in this function */
    QuasarMinidump* minidump = (QuasarMinidump*)context;

    const auto& minidumpFileName = minidump->dumpFileName();
    rename(descriptor.path(), minidumpFileName.c_str());

    const auto& metaFileName = minidump->metaFileName();

    /* rw-rw-rw- */
    const mode_t openMode{S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH};
    const int flags{O_WRONLY | O_CREAT | O_SYNC | O_TRUNC};
    const auto fd = ::open(metaFileName.c_str(), flags, openMode);
    if (fd < 0) {
        return succeeded;
    }
    const auto& meta = minidump->metaData();
    ::write(fd, meta.c_str(), meta.size());

    ::close(fd);

    return succeeded;
}

void QuasarMinidump::init(const std::string& serviceName,
                          const Json::Value& serviceConfig,
                          std::shared_ptr<YandexIO::ITelemetry> telemetry,
                          const std::string& softwareVersion)
{
    if (initialized_.exchange(true)) {
        /**
         * Since QuasarMinidump is a singleton class it should be inited once
         * otherwise race condition may be caused
         */
        throw std::logic_error("QuasarMinidump should be initialized once");
    }
    telemetry_ = std::move(telemetry);
    Y_VERIFY(telemetry_ != nullptr);

    Json::Value minidumpConfig;
    try {
        minidumpConfig = getJson(serviceConfig, "Minidump");
    } catch (const std::runtime_error& error) {
        YIO_LOG_INFO("No minidump configuration for service " << serviceName);
        return;
    }

    enabled_ = tryGetBool(minidumpConfig, "enabled", false);
    if (!enabled_) {
        YIO_LOG_INFO("Minidump is disabled for service " << serviceName);
        return;
    }

    YIO_LOG_INFO("Switch on minidump for service " << serviceName);

    serviceName_ = serviceName;
    dumpFile_ = tryGetString(minidumpConfig, "fileName", serviceName_ + ".dmp");
    metaJsonPath_ = dumpFile_.GetPath() + ".json";
    {
        Json::Value meta;
        meta["softwareVersion"] = softwareVersion;
        metaData_ = jsonToString(meta);
    }

    const auto directory = dumpFile_.Parent();
    YIO_LOG_INFO("Create minidump path " << directory.GetPath());
    try {
        directory.MkDirs();
    } catch (const std::exception& e) {
        enabled_ = false;
        YIO_LOG_ERROR_EVENT("Minidump.MinidumpDirectoryCreateFail", "Can't create minidump directory " << directory.GetPath() << "; error: " << e.what());
        return;
    }

    MinidumpDescriptor descriptor(directory.GetPath());

    auto sizeLimitKB = tryGetInt(minidumpConfig,
                                 "sizeLimitKB",
                                 DEFAULT_SIZE_LIMIT_KB);
    sizeLimitKB = (sizeLimitKB < (int)DEFAULT_SIZE_LIMIT_KB) ? DEFAULT_SIZE_LIMIT_KB : sizeLimitKB;
    bool saveExceptionMessage = tryGetBool(minidumpConfig, "saveExceptionMessage", false);

    YIO_LOG_INFO("Minidump size limit is " << sizeLimitKB << "KB");
    /* Stack size limit is full size limit minus minidump fudge factor */
    descriptor.set_stack_size_limit((sizeLimitKB - LIMIT_MINIDUMP_FUDGE_FACTOR_KB) * KB);

    exceptionHandler_ = std::make_unique<ExceptionHandler>(descriptor,          /* minidump descriptor     */
                                                           nullptr,             /* callback filter         */
                                                           dumpCallback,        /* callback function       */
                                                           this,                /* callback context        */
                                                           true,                /* do install handler      */
                                                           -1,                  /* server descriptor       */
                                                           saveExceptionMessage /* save uncaught exception */
    );

    if (dumpFile_.Exists()) {
        auto moveToMetrica = tryGetBool(minidumpConfig, "moveToMetrica", false);
        if (moveToMetrica) {
            YIO_LOG_INFO("Move previous minidump " << dumpFile_.GetPath() << " to metrica");
            try {
                moveDumpToMetrica();
            } catch (const std::runtime_error& error) {
                YIO_LOG_ERROR_EVENT("Minidump.SendToMetricaFail", "Cannot move minidump " << dumpFile_.GetPath() << " to metrica, error: " << error.what());
            }
        }
    }
}

void QuasarMinidump::moveDumpToMetrica()
{
    auto dump = getFileContent(dumpFile_.GetPath());
    auto compressed = gzipCompress(dump);
    auto base64 = base64Encode(compressed.c_str(), compressed.length());

    Json::Value minidumpEvent;

    minidumpEvent["service"] = serviceName_;
    minidumpEvent["compressedDump"] = base64;

    if (metaJsonPath_.Exists()) {
        try {
            const Json::Value metadataJson = readJsonFromFile(metaJsonPath_.GetPath());
            minidumpEvent["softwareVersion"] = tryGetString(metadataJson, "softwareVersion");
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("Minidump.ReadMetadataFail", e.what());
        }
    }

    telemetry_->reportEvent("minidump", jsonToString(minidumpEvent), true);

    dumpFile_.ForceDelete();
    metaJsonPath_.ForceDelete();
}
