#pragma once

#include <yandex_io/libs/base/persistent_file.h>

#include <json/json.h>

#include <util/folder/path.h>

#include <atomic>
#include <memory>
#include <string>

namespace google_breakpad {
    class ExceptionHandler;
    class MinidumpDescriptor;
} // namespace google_breakpad

namespace YandexIO {
    class ITelemetry;
} // namespace YandexIO

namespace quasar {

    class QuasarMinidump {
    public:
        static QuasarMinidump& getInstance();

        void init(const std::string& serviceName,
                  const Json::Value& serviceConfig,
                  std::shared_ptr<YandexIO::ITelemetry> telemetry,
                  const std::string& softwareVersion);

        std::string serviceName() const {
            return serviceName_;
        }
        const std::string& dumpFileName() const {
            return dumpFile_.GetPath();
        }

        const std::string& metaFileName() const {
            return metaJsonPath_.GetPath();
        }

        const std::string& metaData() const {
            return metaData_;
        }

        bool enabled() const {
            return enabled_;
        }

        QuasarMinidump(const QuasarMinidump&) = delete;
        QuasarMinidump& operator=(const QuasarMinidump&) = delete;

    private:
        std::atomic<bool> initialized_{false};

        std::string serviceName_;
        TFsPath dumpFile_;
        TFsPath metaJsonPath_;
        std::string metaData_;

        std::string softwareVersion_;
        std::shared_ptr<YandexIO::ITelemetry> telemetry_;

        bool enabled_{false};

        std::unique_ptr<google_breakpad::ExceptionHandler> exceptionHandler_{nullptr};

        /**
         * 64KB is minimum space amount for minidump file, because it can fit in the
         * minidump, exclude the stack data, in other words when
         * (stack size limit == 0), see kLimitMinidumpFudgeFactor in the Breakpad
         * for more information.
         */
        static constexpr unsigned int KB = (1U << 10);
        static constexpr unsigned int LIMIT_MINIDUMP_FUDGE_FACTOR_KB = 64U;
        static constexpr unsigned int DEFAULT_SIZE_LIMIT_KB = LIMIT_MINIDUMP_FUDGE_FACTOR_KB;

        static bool dumpCallback(const google_breakpad::MinidumpDescriptor& descriptor, void* context, bool succeeded);

        void moveDumpToMetrica();

        QuasarMinidump() = default;
        ~QuasarMinidump() = default;
    };

} // namespace quasar
