#include "network_interfaces.h"

#include "netlink_monitor.h"

#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/logging/logging.h>

#include <linux/netlink.h>
#include <linux/rtnetlink.h>

#include <algorithm>
#include <memory>
#include <string.h>

#include <netdb.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/types.h>

#include <ifaddrs.h>

using NetworkInterface = quasar::NetworkInterface;

namespace {
    /* getifaddrs implementation */

    struct IfAddrsDeleter {
        void operator()(ifaddrs* ptr) {
            freeifaddrs(ptr);
        }
    };

    using IfAddrsPtr = std::unique_ptr<ifaddrs, IfAddrsDeleter>;

    IfAddrsPtr getInterfaces() {
        ifaddrs* interfaces = nullptr;
        getifaddrs(&interfaces);
        return IfAddrsPtr(interfaces);
    }

    const void* getInAddrFromSockaddr(const sockaddr* addr) {
        if (addr->sa_family == AF_INET) {
            return &(((const sockaddr_in*)addr)->sin_addr);
        }
        if (addr->sa_family == AF_INET6) {
            return &(((const sockaddr_in6*)addr)->sin6_addr);
        }
        return nullptr;
    }

    std::string sockaddrToString(const sockaddr* addr) {
        if (!addr) {
            return {};
        }
        char buf[1024];
        auto str = inet_ntop(addr->sa_family, getInAddrFromSockaddr(addr), buf, sizeof(buf));
        if (str) {
            return str;
        }
        return {};
    }

    bool isActiveFlags(unsigned int flags) {
        return ((flags & (IFF_UP | IFF_RUNNING)) && !(flags & IFF_LOOPBACK));
    }

    NetworkInterface::Family convertFamily(sa_family_t src) {
        using Family = NetworkInterface::Family;
        switch (src) {
            case AF_INET:
                return Family::IPV4;
            case AF_INET6:
                return Family::IPV6;
            case AF_PACKET:
                return Family::PACKET;
        }
        return Family::UNSPEC;
    }

    using NetlinkMonitor = quasar::net::NetlinkMonitor;

    struct InterfacesBuilder: public NetlinkMonitor::Handler {
        using IfFlags = NetlinkMonitor::IfFlags;
        using Scope = NetlinkMonitor::Scope;

        using AddrFamily = NetworkInterface::Family;

        struct Addr {
            std::string addr;
            bool local = false;
            bool link = false;
            AddrFamily family;
        };

        struct Interface {
            std::string name;
            std::string mac;
            IfFlags flags;
            std::vector<Addr> addrs;
        };

        std::unordered_map<IfIndex, Interface> links;

        static AddrFamily toAddrFamily(Family family) {
            switch (family) {
                case AF_INET:
                    return AddrFamily::IPV4;
                case AF_INET6:
                    return AddrFamily::IPV6;
                case AF_PACKET:
                    return AddrFamily::PACKET;
                default:;
            }
            return AddrFamily::UNSPEC;
        }

        void onStats(IfIndex /*idx*/, const Stats& /*stats*/) override {
        }

        void onAddressRemove(IfIndex /*idx*/, std::string /*addr*/, Family /*family*/, Scope /*scope*/, bool /*local*/) override {
        }

        void onAddress(IfIndex idx, std::string addr, Family family, Scope /*scope*/, bool local) override {
            auto& addrs = links[idx].addrs;
            auto addrFamily = toAddrFamily(family);
            auto iter = std::find_if(
                std::begin(addrs), std::end(addrs),
                [&addr, addrFamily](const Addr& a) -> bool {
                    return a.addr == addr && a.family == addrFamily;
                });
            if (iter == std::end(addrs)) {
                addrs.emplace_back(Addr{.addr = std::move(addr), .local = local, .link = !local, .family = addrFamily});
            } else {
                if (local) {
                    iter->local = true;
                } else {
                    iter->link = true;
                }
            }
        }

        void onLink(IfIndex idx, std::string name, IfFlags flags) override {
            auto& iface = links[idx];
            iface.name = std::move(name);
            iface.flags = flags;
        }

        void onLinkRemoved(IfIndex /*idx*/) override{};

        void onMac(IfIndex idx, std::string mac) override {
            links[idx].mac = std::move(mac);
        }
    };

} // namespace

namespace quasar {

    NetworkInterface::NetworkInterface(std::string inName, std::string inAddress, Family f)
        : name(std::move(inName))
        , address(std::move(inAddress))
        , family(f)
    {
    }

    std::vector<NetworkInterface> getActiveNetworkInterfacesIfAddrs() {
        auto interfaces = getInterfaces();
        YIO_LOG_DEBUG("Number of interfaces = " << interfaces.get());
        std::vector<NetworkInterface> result;
        if (interfaces) {
            auto ptr = interfaces.get();
            while (ptr) {
                if (isActiveFlags(ptr->ifa_flags)) {
                    std::string strAddr = sockaddrToString(ptr->ifa_addr);
                    if (!strAddr.empty()) {
                        result.emplace_back(ptr->ifa_name, std::move(strAddr), convertFamily(ptr->ifa_addr->sa_family));
                    }
                }
                ptr = ptr->ifa_next;
            }
        } else {
            YIO_LOG_INFO("Cannot get interfaces!");
        }
        return result;
    }

    std::vector<NetworkInterface> getActiveNetworkInterfaces() {
        std::vector<NetworkInterface> result;
        try {
            InterfacesBuilder builder;
            auto netlinkReqs = quasar::net::makeNetlinkMonitor(builder);
            netlinkReqs->monitor(false);

            for (const auto& [idx, dev] : builder.links) {
                if (!dev.flags.up || !dev.flags.running || dev.flags.loopback) {
                    continue;
                }
                for (const auto& addr : dev.addrs) {
                    if (addr.link) {
                        result.emplace_back(dev.name, addr.addr, addr.family);
                    }
                }
            }
        } catch (const std::runtime_error& e) {
            YIO_LOG_DEBUG(e.what());
        } catch (...) {
            YIO_LOG_DEBUG("Uknown exception during netlik iteration");
        }
        return result;
    }

    std::optional<std::string> getLocalAddressAsOptString() {
        auto interfaces = quasar::getActiveNetworkInterfaces();
        using Family = NetworkInterface::Family;
        for (const auto& iface : interfaces) {
            if (iface.name.starts_with("usb")) {
                continue;
            }
            if (iface.family == Family::IPV4) {
                return iface.address;
            }
        }
        if (!interfaces.empty()) {
            return interfaces.front().address;
        }
        return {};
    }

    std::string getLocalAddress() {
        auto address = getLocalAddressAsOptString();
        if (address) {
            return std::move(*address);
        }
        return "";
    }

} // namespace quasar
