#include "ntp_client.h"

#include "ntp_client_api.h"
#include "ntp_client_exception.h"
#include "ntp_packet_raw.h"

#include <algorithm>
#include <chrono>
#include <cstring>
#include <iostream>
#include <memory>
#include <random>
#include <set>
#include <stdexcept>
#include <string.h>
#include <type_traits>
#include <unordered_map>

#include <netdb.h>
#include <poll.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/types.h>

namespace quasar {

    namespace {

        using AddrInfoPtr = std::shared_ptr<struct addrinfo>;

        std::shared_ptr<NtpClientApi> defaultApi()
        {
            std::shared_ptr<NtpClientApi> api = std::make_shared<NtpClientApi>();
            api->getaddrinfo = getaddrinfo;
            api->freeaddrinfo = freeaddrinfo;
            api->socket = socket;
            api->setsockopt = setsockopt;
            api->close = close;
            api->inet_ntop = inet_ntop;
            api->sendto = sendto;
            api->recvfrom = recvfrom;
            api->poll = poll;

            return api;
        }

        struct Socket {
            int fd;
            struct sockaddr* aiAddr;

            std::string hostName;
            std::string ipAddress;
            uint16_t port;

            std::shared_ptr<void> resourceGuard;
        };
        using SocketPtr = std::shared_ptr<Socket>;

        std::vector<AddrInfoPtr> getAddrInfo(const NtpClientApi& api, const std::string& host, uint16_t port)
        {
            std::string portText = std::to_string(port);
            struct addrinfo hints;
            struct addrinfo* res = nullptr;

            std::memset(&hints, 0, sizeof(hints));
            hints.ai_socktype = SOCK_DGRAM;

            auto errorCode = api.getaddrinfo(host.c_str(), portText.c_str(), &hints, &res);
            if (errorCode != 0) {
                int errnum = (errorCode == EAI_SYSTEM ? errno : 0);
                throw NtpClientExceptionAddrInfo(host, port, errorCode, errnum);
            }

            auto apiFreeaddrinfo = api.freeaddrinfo;
            size_t count = 0;
            for (auto n = res; n; n = n->ai_next) {
                ++count;
            }
            std::vector<AddrInfoPtr> result;
            result.reserve(count);
            for (auto n = res; n; n = n->ai_next) {
                if (result.empty()) {
                    result.emplace_back(AddrInfoPtr(n, [apiFreeaddrinfo](struct addrinfo* p) { apiFreeaddrinfo(p); }));
                } else {
                    result.emplace_back(AddrInfoPtr(result.front(), n));
                }
            }
            return result;
        }

        SocketPtr openSocket(const NtpClientApi& api, const NtpClient::Addr& addr, const AddrInfoPtr& addrInfo, std::chrono::milliseconds timeout)
        {
            for (auto ap = addrInfo.get(); ap; ap = ap->ai_next) {
                int fd = api.socket(ap->ai_family, ap->ai_socktype, ap->ai_protocol);
                if (fd != -1) {
                    struct timeval tv {
                        static_cast<time_t>(timeout.count() / 1000),
                            static_cast<suseconds_t>(timeout.count() % 1000 * 1000)
                    };
                    if (api.setsockopt(fd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv)) < 0) {
                        api.close(fd);
                        continue;
                    }
                    auto apiClose = api.close;
                    std::string ipAddress;
                    if (ap->ai_addr->sa_family == AF_INET) {
                        char ipv4[INET_ADDRSTRLEN];
                        struct sockaddr_in* addr4;
                        addr4 = (struct sockaddr_in*)ap->ai_addr;
                        api.inet_ntop(AF_INET, &addr4->sin_addr, ipv4, INET_ADDRSTRLEN);
                        ipAddress = ipv4;
                    } else if (ap->ai_addr->sa_family == AF_INET6) {
                        char ipv6[INET6_ADDRSTRLEN];
                        struct sockaddr_in6* addr6;
                        addr6 = (struct sockaddr_in6*)ap->ai_addr;
                        api.inet_ntop(AF_INET6, &addr6->sin6_addr, ipv6, INET6_ADDRSTRLEN);
                        ipAddress = ipv6;
                    }
                    return SocketPtr(new Socket{fd, ap->ai_addr, addr.host, ipAddress, static_cast<uint16_t>(ap->ai_protocol), addrInfo}, [apiClose](Socket* p) { apiClose(p->fd); delete p; });
                }
            }
            throw std::runtime_error("Can't connect to address");
        }

        class AbortedByUser: public std::runtime_error {
        public:
            AbortedByUser()
                : std::runtime_error("Aborted by user")
            {
            }
        };

        class NtpPacket {
        public:
            using duration = std::chrono::steady_clock::duration;
            using time_point = std::chrono::steady_clock::time_point;
            using Raw = NtpPacketRaw;

            NtpPacket()
            {
                static_assert(sizeof(Raw) == 12 * sizeof(uint32_t), "Unexpected NTP data packet size of alignment");
                static_assert(std::is_pod<Raw>::value, "Raw struct must be POD type");
                std::memset(&raw_, 0, sizeof(raw_));
            }

            NtpPacket(Raw raw)
                : raw_(raw)
            {
            }

            uint8_t leap() const {
                return raw_.flags >> 6;
            }
            uint8_t version() const {
                return ((raw_.flags & 0x3f) >> 3);
            }
            uint8_t mode() const {
                return raw_.flags & 0x07;
            }
            uint8_t stratum() const {
                return raw_.stratum;
            }
            uint8_t poll() const {
                return raw_.poll;
            }
            uint8_t precision() const {
                return raw_.precision;
            }
            uint32_t rootDelay() const {
                return fixEndian(raw_.root_delay);
            }
            uint32_t rootDispersion() const {
                return fixEndian(raw_.root_dispersion);
            }
            uint32_t referenceId() const {
                uint32_t value;
                static_assert(sizeof(value) == sizeof(raw_.referenceId), "invalid sizeof");
                std::memcpy(&value, &raw_.referenceId, sizeof(value));
                return value;
            }
            std::string referenceIdText() const {
                char id[9];
                if (raw_.referenceId[0] >= 'A' && raw_.referenceId[0] <= 'Z' &&
                    raw_.referenceId[1] >= 'A' && raw_.referenceId[1] <= 'Z' &&
                    raw_.referenceId[2] >= 'A' && raw_.referenceId[2] <= 'Z' &&
                    (raw_.referenceId[3] >= 'A' && raw_.referenceId[3] <= 'Z' || raw_.referenceId[3] == 0))
                {
                    std::memcpy(id, raw_.referenceId, 4);
                    id[4] = 0;
                } else {
                    const char* hex = "01234567890ABCDEF";
                    for (size_t i = 0; i < 4; ++i) {
                        id[i * 2] = hex[raw_.referenceId[i] >> 4];
                        id[i * 2 + 1] = hex[raw_.referenceId[i] & 0xf];
                    }
                    id[8] = 0;
                }
                return id;
            }
            time_point referenceTime() const {
                return time_point{toUnixTimestampEpoch(toMicroseconds(raw_.ref_ts_sec, raw_.ref_ts_frac))};
            }
            time_point originTime() const {
                return time_point{toMicroseconds(raw_.origin_ts_sec, raw_.origin_ts_frac)};
            }
            time_point receiveTime() const {
                return time_point{toUnixTimestampEpoch(toMicroseconds(raw_.recv_ts_sec, raw_.recv_ts_frac))};
            }
            time_point transmitTime() const {
                return time_point{toUnixTimestampEpoch(toMicroseconds(raw_.trans_ts_sec, raw_.trans_ts_frac))};
            }
            time_point destinationTime() const {
                return destinationTime_;
            }

            operator bool() const {
                return !!*this;
            }

            bool operator!() const {
                const uint8_t* ptr = (const uint8_t*)&raw_;
                for (size_t i = 0; i < sizeof(raw_); i++) {
                    if (ptr[i]) {
                        return false;
                    }
                }
                return true;
            }
            void stampOriginTimeIfZero(const NtpPacket& other)
            {
                if (raw_.origin_ts_sec == 0 && raw_.origin_ts_frac == 0) {
                    raw_.origin_ts_sec = other.raw_.origin_ts_sec;
                    raw_.origin_ts_frac = other.raw_.origin_ts_frac;
                }
            }
            void stampDestinationTime()
            {
                destinationTime_ = now();
            }

            size_t dataSize() const {
                return sizeof(raw_);
            }
            const void* data() const {
                return &raw_;
            }
            void* data() {
                return &raw_;
            }

            bool isUnsynchronized() const {
                return leap() == 3 || stratum() == 16;
            }

            bool isKissOfDeath() const {
                return stratum() == 0;
            }

            static time_point now()
            {
                return std::chrono::steady_clock::now();
            }

            static NtpPacket makeClientHello()
            {
                Raw raw;
                std::memset(&raw, 0, sizeof(raw));
                raw.flags = 0xE3; // LI=11b, VN=100b, Mode=011b
                auto origin = makeNtpTime(now().time_since_epoch());
                raw.origin_ts_sec = origin.first;
                raw.origin_ts_frac = origin.second;
                return NtpPacket(raw);
            }

        private:
            static std::pair<uint32_t, uint32_t> makeNtpTime(std::chrono::nanoseconds ns)
            {
                uint64_t num = ns.count() / 1000000000ull;
                uint64_t denon = ns.count() % 1000000000ull;

                constexpr double k = static_cast<double>(0xFFFFFFFF) / 1000000000.;
                return {fixEndian(static_cast<uint32_t>(num)), fixEndian(static_cast<uint32_t>(k * denon))};
            }

            static uint32_t fixEndian(uint32_t value)
            {
                return (value >> 24) | ((value & 0x00ff0000) >> 8) | ((value & 0x0000ff00) << 8) | ((value & 0x000000ff) << 24);
            }

            static std::chrono::microseconds toMicroseconds(uint64_t seconds, uint64_t frac)
            {
                uint64_t umsFrac = (static_cast<double>(fixEndian(frac)) * 1000000ull) / static_cast<double>(0xFFFFFFFFull);
                return std::chrono::microseconds{fixEndian(seconds) * 1000000ull + umsFrac};
            }

            static std::chrono::microseconds toUnixTimestampEpoch(std::chrono::microseconds epoch1900)
            {
                // https://tools.ietf.org/html/rfc5905#appendix-A.4
                return epoch1900 - std::chrono::microseconds{2208988800000000ull};
            }

        private:
            Raw raw_;
            time_point destinationTime_;
        };

        std::pair<NtpClient::ServerTimeCredibility, NtpPacket::duration> calculateDiff(std::vector<NtpPacket> ntpPackets)
        {
            if (ntpPackets.empty()) {
                return {NtpClient::ServerTimeCredibility::UNDEFINED, std::chrono::nanoseconds{0}};
            }

            // Here we need to think about the reliability of measurements and throw too much variance an exception.
            // For now, take N count with the best latency.
            constexpr size_t mean = 10;
            NtpClient::ServerTimeCredibility serverTimeCredibility = NtpClient::ServerTimeCredibility::APPROXIMATE;
            std::sort(ntpPackets.begin(), ntpPackets.end(),
                      [](const NtpPacket& p1, const NtpPacket& p2) {
                          auto latency1 = p1.destinationTime() - p1.originTime();
                          auto latency2 = p2.destinationTime() - p2.originTime();
                          return latency1 < latency2;
                      });
            if (ntpPackets.size() >= mean) {
                ntpPackets.resize(mean);
                serverTimeCredibility = NtpClient::ServerTimeCredibility::RELIABLY;
            }

            double diffNs = 0.;
            for (const NtpPacket& p : ntpPackets) {
                diffNs += std::chrono::duration_cast<std::chrono::nanoseconds>(p.receiveTime().time_since_epoch()).count();
                diffNs -= std::chrono::duration_cast<std::chrono::nanoseconds>(p.originTime().time_since_epoch()).count();
                diffNs += std::chrono::duration_cast<std::chrono::nanoseconds>(p.transmitTime().time_since_epoch()).count();
                diffNs -= std::chrono::duration_cast<std::chrono::nanoseconds>(p.destinationTime().time_since_epoch()).count();
            }
            diffNs /= ntpPackets.size() * 2;

            NtpPacket::duration diff = std::chrono::duration_cast<NtpPacket::duration>(std::chrono::nanoseconds{static_cast<int64_t>(diffNs)});
            return {serverTimeCredibility, diff};
        }

    } // namespace

    struct NtpClient::DiscoveryResult {
        Addr addr;
        std::string ipAddress;
        AddrInfoPtr addrInfo;
        NtpPacket ntpPacket;
    };

    NtpClient::Addr::Addr(std::string inHost)
    {
        std::size_t found = inHost.rfind(':');
        if (found != std::string::npos) {
            auto it = inHost.begin() + found;
            host = std::string(inHost.begin(), it);
            ++it;
            try {
                port = std::stoi(std::string(it, inHost.end()));
            } catch (const std::exception& ex) {
                throw std::invalid_argument(std::string("NtpClient: invalid port number: ") + ex.what());
            }
        } else {
            host = std::move(inHost);
        }
    }

    NtpClient::Addr::Addr(std::string inHost, uint16_t inPort)
        : host(std::move(inHost))
        , port(inPort)
    {
    }

    bool NtpClient::Addr::operator<(const Addr& other) const {
        return (host != other.host ? host < other.host : port < other.port);
    }

    bool NtpClient::Addr::operator==(const Addr& other) const {
        return host == other.host && port == other.port;
    }

    bool NtpClient::Addr::operator!=(const Addr& other) const {
        return !(*this == other);
    }

    void NtpClient::Params::parseJson(const Json::Value& config)
    {
        Params backup = *this;
        try {
            if (config.isMember("timeoutMs")) {
                timeout = std::chrono::milliseconds{getUInt64(config, "timeoutMs")};
            }

            if (config.isMember("minMeasuringCount")) {
                minMeasuringCount = getUInt64(config, "minMeasuringCount");
            }

            if (config.isMember("sufficientMeasuringCount")) {
                sufficientMeasuringCount = getUInt64(config, "sufficientMeasuringCount");
            }

            if (config.isMember("maxMeasuringCount")) {
                maxMeasuringCount = getUInt64(config, "maxMeasuringCount");
            }

            if (config.isMember("ntpServers")) {
                const Json::Value& jNtpServers = config["ntpServers"];
                if (jNtpServers.isArray()) {
                    ntpServers.clear();
                    for (const auto& jNtpServer : jNtpServers) {
                        if (jNtpServer.isString()) {
                            ntpServers.emplace_back(Addr(jNtpServer.asString()));
                        } else if (jNtpServer.isObject()) {
                            std::string host = getString(jNtpServer, "host");
                            int64_t port = getInt64(jNtpServer, "port");
                            if (port <= 0 || port >= 0xFFFF) {
                                throw std::invalid_argument("Fail to parse ntp json config, port value is invalid");
                            }
                            ntpServers.emplace_back(NtpClient::Addr{host, static_cast<uint16_t>(port)});
                        }
                    }
                } else {
                    throw std::invalid_argument("Invalid json config: ntpServers must be an array");
                }
            }
        } catch (...) {
            *this = backup;
            throw;
        }
    }

    bool NtpClient::Params::tryParseJson(const Json::Value& config) noexcept {
        try {
            parseJson(config);
        } catch (...) {
            return false;
        }
        return true;
    }

    Json::Value NtpClient::Params::toJson() const noexcept {
        Json::Value json;
        json["timeoutMs"] = static_cast<int64_t>(timeout.count());
        json["minMeasuringCount"] = static_cast<int64_t>(minMeasuringCount);
        json["sufficientMeasuringCount"] = static_cast<int64_t>(sufficientMeasuringCount);
        json["maxMeasuringCount"] = static_cast<int64_t>(maxMeasuringCount);
        for (const auto& addr : ntpServers) {
            if (addr.port == 123) {
                json["ntpServers"].append(addr.host);
            } else {
                Json::Value v;
                v["host"] = addr.host;
                v["port"] = addr.port;
                json["ntpServers"].append(std::move(v));
            }
        }
        return json;
    }

    bool NtpClient::Params::operator==(const Params& other) const {
        return timeout == other.timeout &&
               minMeasuringCount == other.minMeasuringCount &&
               sufficientMeasuringCount == other.sufficientMeasuringCount &&
               maxMeasuringCount == other.maxMeasuringCount &&
               ntpServers == other.ntpServers;
    }

    bool NtpClient::Params::operator!=(const Params& other) const {
        return !(*this == other);
    }

    NtpClient::NtpResult::NtpResult()
        : serverTimeCredibility(ServerTimeCredibility::UNDEFINED)
        , diff{0}
    {
    }

    NtpClient::NtpResult::NtpResult(SyncMode inSyncMode, std::vector<Addr> inResolvedIps, NtpClient::Addr intAddr, std::string inIpAddress, std::string inReferenceId, ServerTimeCredibility inServerTimeCredibility, NtpClient::duration inDiff)
        : syncMode(inSyncMode)
        , resolvedIps(std::move(inResolvedIps))
        , addr(std::move(intAddr))
        , ipAddress(std::move(inIpAddress))
        , referenceId(std::move(inReferenceId))
        , serverTimeCredibility(inServerTimeCredibility)
        , diff(inDiff)
    {
    }

    NtpClient::time_point NtpClient::NtpResult::syncTime() const {
        return std::chrono::steady_clock::now() + diff;
    }

    NtpClient::time_point NtpClient::NtpResult::systemTime()
    {
        std::chrono::nanoseconds d = std::chrono::duration_cast<std::chrono::nanoseconds>(std::chrono::system_clock::now().time_since_epoch());
        return NtpClient::time_point{d};
    }

    std::chrono::nanoseconds NtpClient::NtpResult::syncDiff() const {
        auto systemTimestamp = std::chrono::duration_cast<std::chrono::nanoseconds>(std::chrono::system_clock::now().time_since_epoch()).count();
        auto serverTimestamp = std::chrono::duration_cast<std::chrono::nanoseconds>(syncTime().time_since_epoch()).count();
        return std::chrono::nanoseconds{serverTimestamp - systemTimestamp};
    }

    std::chrono::nanoseconds NtpClient::NtpResult::absSyncDiff() const {
        return std::chrono::nanoseconds{std::abs(syncDiff().count())};
    }

    NtpClient::NtpClient(Params params)
        : NtpClient(std::move(params), defaultApi())
    {
    }

    NtpClient::NtpClient(Params params, std::shared_ptr<const NtpClientApi> api)
        : api_(std::move(api))
        , params_(std::move(params))
    {
        if (params_.timeout == std::chrono::milliseconds{0}) {
            params_.timeout = std::chrono::milliseconds{1000};
        }

        if (params_.ntpServers.empty()) {
            params_.ntpServers.emplace_back(Addr{std::string("pool.ntp.org"), 123});
        }

        params_.minMeasuringCount = std::max<size_t>(params_.minMeasuringCount, 1);
        params_.maxMeasuringCount = std::max<size_t>(params_.minMeasuringCount, params_.maxMeasuringCount);
        params_.maxMeasuringCount = std::max<size_t>(params_.sufficientMeasuringCount, params_.maxMeasuringCount);
    }

    NtpClient::NtpResult NtpClient::sync(SyncMode syncMode) const {
        std::atomic<bool> abortFlag{false};
        return sync(syncMode, abortFlag);
    }

    NtpClient::NtpResult NtpClient::sync(SyncMode syncMode, const std::atomic<bool>& abortFlag) const {
        struct ClientRequest {
            Addr addr;
            AddrInfoPtr addrInfo;
        };
        std::vector<ClientRequest> clientRequests;
        std::set<Addr> resolvedIps;

        switch (syncMode)
        {
            case SyncMode::DEFAULT:
            case SyncMode::RANDOMIZE:
                clientRequests.reserve(params_.ntpServers.size());
                for (const auto& ntpServer : params_.ntpServers) {
                    clientRequests.emplace_back(ClientRequest{ntpServer, nullptr});
                }
                if (syncMode == SyncMode::RANDOMIZE) {
                    std::shuffle(clientRequests.begin(), clientRequests.end(),
                                 std::default_random_engine(std::chrono::system_clock::now().time_since_epoch().count()));
                }
                break;
            case SyncMode::DISCOVERY: {
                std::vector<DiscoveryResultPtr> discoveryResults = discoveryRequest(params_.timeout, params_.sufficientMeasuringCount == 1 ? 1 : 0xFFFFFF, abortFlag);
                for (const auto& d : discoveryResults) {
                    resolvedIps.emplace(d->ipAddress, d->addr.port);
                }
                if (discoveryResults.empty()) {
                    throw NtpClientException("NtpClient: Fail to connect to any ntp servers");
                } else if (params_.sufficientMeasuringCount == 1) {
                    DiscoveryResult& discoveryResult = *discoveryResults.front();
                    auto diff = calculateDiff({discoveryResult.ntpPacket});
                    return NtpResult{
                        SyncMode::DISCOVERY,
                        std::vector<Addr>(resolvedIps.begin(), resolvedIps.end()),
                        std::move(discoveryResult.addr),
                        std::move(discoveryResult.ipAddress),
                        discoveryResult.ntpPacket.referenceIdText(),
                        diff.first,
                        diff.second,
                    };
                }

                clientRequests.reserve(discoveryResults.size());
                for (const auto& discoveryResult : discoveryResults) {
                    clientRequests.emplace_back(ClientRequest{
                        std::move(discoveryResult->addr),
                        std::move(discoveryResult->addrInfo),
                    });
                }
            } break;
            case SyncMode::CUSTOM:
                return NtpClient::NtpResult{};
        }

        std::function<void()> emitException;
        int errorCode = 0;
        for (const auto& clientRequest : clientRequests) {
            const Addr& addr = clientRequest.addr;
            try {
                std::vector<AddrInfoPtr> addrInfos;
                if (clientRequest.addrInfo) {
                    addrInfos.push_back(clientRequest.addrInfo);
                } else {
                    addrInfos = getAddrInfo(*api_, addr.host, addr.port);
                }
                for (const auto& addrInfo : addrInfos) {
                    std::string ipAddress;
                    try {
                        SocketPtr serverSocket = openSocket(*api_, addr, addrInfo, params_.timeout);
                        socklen_t addrlen = sizeof(struct sockaddr_storage);
                        ipAddress = serverSocket->ipAddress;
                        resolvedIps.emplace(ipAddress, addr.port);

                        std::unordered_map<int, size_t> sendtoErrors;
                        std::unordered_map<int, size_t> recvfromErrors;
                        std::vector<NtpPacket> ntpPackets;

                        ntpPackets.reserve(params_.sufficientMeasuringCount);
                        sendtoErrors.reserve(params_.maxMeasuringCount);
                        recvfromErrors.reserve(params_.maxMeasuringCount);

                        for (size_t measuringNumber = 0; measuringNumber < params_.maxMeasuringCount && ntpPackets.size() < params_.sufficientMeasuringCount; ++measuringNumber) {
                            NtpPacket replNtpPacket;
                            NtpPacket heloNtpPacket = NtpPacket::makeClientHello();
                            errorCode = api_->sendto(serverSocket->fd, heloNtpPacket.data(), heloNtpPacket.dataSize(), 0, serverSocket->aiAddr, addrlen);
                            if (errorCode == -1) {
                                ++sendtoErrors[errno];
                                continue;
                            }

                            errorCode = api_->recvfrom(serverSocket->fd, replNtpPacket.data(), replNtpPacket.dataSize(), 0, serverSocket->aiAddr, &addrlen);
                            if (errorCode == -1) {
                                ++recvfromErrors[errno];
                                continue;
                            }
                            replNtpPacket.stampDestinationTime();
                            replNtpPacket.stampOriginTimeIfZero(heloNtpPacket);
                            if (replNtpPacket.originTime() != heloNtpPacket.originTime()) {
                                throw NtpClientExceptionTimeForgery(addr.host, addr.port, serverSocket->ipAddress);
                            }
                            if (replNtpPacket.isUnsynchronized()) {
                                throw NtpClientExceptionClockUnsynchronized(addr.host, addr.port, serverSocket->ipAddress);
                            }
                            if (replNtpPacket.isKissOfDeath()) {
                                throw NtpClientExceptionKissOfDeath(replNtpPacket.referenceIdText(), addr.host, addr.port, serverSocket->ipAddress);
                            }
                            ntpPackets.emplace_back(replNtpPacket);

                            if (abortFlag) {
                                throw AbortedByUser{};
                            }
                        }

                        if (ntpPackets.empty() || ntpPackets.size() < params_.minMeasuringCount) {
                            if (ntpPackets.empty()) {
                                throw std::runtime_error("Failed to get any data");
                            } else {
                                throw std::runtime_error("Not enough data for reliable synchronization");
                            }
                        }

                        auto diff = calculateDiff(ntpPackets);
                        NtpResult result{
                            syncMode,
                            std::vector<Addr>(resolvedIps.begin(), resolvedIps.end()),
                            addr,
                            serverSocket->ipAddress,
                            ntpPackets.back().referenceIdText(),
                            diff.first,
                            diff.second,
                        };

                        constexpr int64_t memorableDate = 1577836800ull; // 2020-01-01
                        if (std::chrono::duration_cast<std::chrono::seconds>(result.syncTime().time_since_epoch()).count() < memorableDate) {
                            throw NtpClientExceptionWrongTime(addr.host, addr.port, serverSocket->ipAddress);
                        }
                        return result;
                    } catch (const AbortedByUser&) {
                        throw;
                    } catch (const NtpClientException&) {
                        auto eptr = std::current_exception();
                        emitException = [eptr] { std::rethrow_exception(eptr); };
                    } catch (const std::exception& e) {
                        std::string message = e.what();
                        emitException = [message = std::move(message), addr, ipAddress] { throw NtpClientException(message, addr.host, addr.port, ipAddress); };
                    } catch (...) {
                        if (!emitException) {
                            emitException = [addr, ipAddress] { throw NtpClientException("Undefined exception", addr.host, addr.port, ipAddress); };
                        }
                    }
                }
            } catch (const AbortedByUser&) {
                throw;
            } catch (const NtpClientException&) {
                auto exptr = std::current_exception();
                emitException = [exptr] { std::rethrow_exception(exptr); };
            } catch (const std::exception& e) {
                auto message = e.what();
                emitException = [message, addr] { throw NtpClientException(message, addr.host, addr.port, ""); };
            } catch (...) {
                if (!emitException) {
                    emitException = [addr] { throw NtpClientException("Undefined exception", addr.host, addr.port, ""); };
                }
            }
        }
        if (!emitException) {
            emitException = [] { throw NtpClientException("No ntp servers"); };
        }
        emitException();
        return {};
    }

    std::vector<NtpClient::DiscoveryResultPtr> NtpClient::discoveryRequest(std::chrono::milliseconds discoveryTimeoutMs, size_t discoveryCount, const std::atomic<bool>& abortFlag) const {
        const auto seed = std::chrono::system_clock::now().time_since_epoch().count();
        std::default_random_engine rndgen{static_cast<std::default_random_engine::result_type>(seed)};
        std::vector<Addr> ntpServers(params_.ntpServers);
        std::shuffle(ntpServers.begin(), ntpServers.end(), rndgen);

        std::vector<NtpClientExceptionDiscoveryFail::Host> discoveryErrors;

        int errorCode = 0;
        discoveryCount = std::max<size_t>(discoveryCount, 1);

        /* Resolve host name */
        std::vector<std::pair<const Addr*, AddrInfoPtr>> addrInfos;
        addrInfos.reserve(ntpServers.size() * 8);
        for (const auto& addr : ntpServers) {
            try {
                if (abortFlag) {
                    throw AbortedByUser{};
                }
                auto res = getAddrInfo(*api_, addr.host, addr.port);
                for (const auto& r : res) {
                    addrInfos.emplace_back(std::make_pair(&addr, r));
                }
            } catch (const NtpClientExceptionAddrInfo& ex) {
                discoveryErrors.emplace_back(NtpClientExceptionDiscoveryFail::Host{ex.host, ex.port, "", NtpClientExceptionDiscoveryFail::ErrorId::GETADDRINFO, ex.errnum, ex.eaiError});
            } catch (...) {
                discoveryErrors.emplace_back(NtpClientExceptionDiscoveryFail::Host{addr.host, addr.port, "", NtpClientExceptionDiscoveryFail::ErrorId::GETADDRINFO});
            }
        }
        std::shuffle(addrInfos.begin(), addrInfos.end(), rndgen);

        struct DiscoveryRequest {
            Addr addr;
            std::string ipAddress;
            AddrInfoPtr addrInfo;
            SocketPtr socket;
            NtpPacket heloNtpPacket;
        };
        std::vector<DiscoveryRequest> discoveryRequests;
        discoveryRequests.reserve(addrInfos.size());

        for (size_t i = 0; i < addrInfos.size(); ++i) {
            const auto& addr = *addrInfos[i].first;
            const auto& addrInfo = addrInfos[i].second;
            if (!addrInfo) {
                continue;
            }
            SocketPtr socket;
            std::string ipAddress;
            try {
                SocketPtr testSocket = openSocket(*api_, addr, addrInfo, discoveryTimeoutMs);
                socklen_t addrlen = sizeof(struct sockaddr_storage);

                ipAddress = testSocket->ipAddress;
                NtpPacket heloNtpPacket = NtpPacket::makeClientHello();
                errorCode = api_->sendto(testSocket->fd, heloNtpPacket.data(), heloNtpPacket.dataSize(), 0, testSocket->aiAddr, addrlen);
                if (errorCode == -1) {
                    discoveryErrors.emplace_back(NtpClientExceptionDiscoveryFail::Host{addr.host, addr.port, ipAddress, NtpClientExceptionDiscoveryFail::ErrorId::SEND, errno});
                    continue;
                }
                discoveryRequests.emplace_back(DiscoveryRequest{addr, ipAddress, addrInfo, testSocket, heloNtpPacket});
            } catch (...) {
                discoveryErrors.emplace_back(NtpClientExceptionDiscoveryFail::Host{addr.host, addr.port, ipAddress, NtpClientExceptionDiscoveryFail::ErrorId::OTHER, 0});
            }
        }

        if (discoveryRequests.empty()) {
            throw NtpClientExceptionDiscoveryFail(discoveryErrors);
        }

        std::vector<DiscoveryResultPtr> discoveryResults;
        discoveryResults.reserve(discoveryRequests.size());

        std::vector<pollfd> rset;
        rset.reserve(discoveryRequests.size());
        for (; discoveryCount;) {
            rset.clear();

            for (auto& discoveryRequest : discoveryRequests) {
                if (!discoveryRequest.socket) {
                    continue;
                }
                rset.emplace_back(pollfd{discoveryRequest.socket->fd, POLLIN | POLLPRI | POLLERR, 0});
            }
            if (rset.empty()) {
                break;
            }

            int nready = api_->poll(rset.data(), rset.size(), discoveryTimeoutMs.count());
            if (nready <= 0) {
                for (auto& discoveryRequest : discoveryRequests) {
                    if (!discoveryRequest.socket) {
                        continue;
                    }
                    discoveryErrors.emplace_back(NtpClientExceptionDiscoveryFail::Host{
                        discoveryRequest.addr.host, discoveryRequest.addr.port, discoveryRequest.ipAddress, nready == 0 ? NtpClientExceptionDiscoveryFail::ErrorId::TIMEOUT : NtpClientExceptionDiscoveryFail::ErrorId::OTHER, 0,
                    });
                }
                break;
            }

            auto it = discoveryRequests.begin();
            for (const auto& r : rset) {
                if (r.revents & (POLLIN | POLLPRI)) {
                    for (; it != discoveryRequests.end(); ++it) {
                        auto& discoveryRequest = *it;
                        if (discoveryRequest.socket && r.fd == discoveryRequest.socket->fd) {
                            socklen_t addrlen = sizeof(struct sockaddr_storage);
                            NtpPacket replNtpPacket;
                            auto errorCode = api_->recvfrom(discoveryRequest.socket->fd, replNtpPacket.data(), replNtpPacket.dataSize(), 0, discoveryRequest.socket->aiAddr, &addrlen);
                            if (errorCode != -1) {
                                replNtpPacket.stampDestinationTime();
                                replNtpPacket.stampOriginTimeIfZero(discoveryRequest.heloNtpPacket);

                                if (replNtpPacket.originTime() != discoveryRequest.heloNtpPacket.originTime()) {
                                    discoveryErrors.emplace_back(NtpClientExceptionDiscoveryFail::Host{
                                        discoveryRequest.addr.host, discoveryRequest.addr.port, discoveryRequest.ipAddress, NtpClientExceptionDiscoveryFail::ErrorId::NTP_TIME_FORGERY, 0,
                                    });
                                } else if (replNtpPacket.isUnsynchronized()) {
                                    discoveryErrors.emplace_back(NtpClientExceptionDiscoveryFail::Host{
                                        discoveryRequest.addr.host, discoveryRequest.addr.port, discoveryRequest.ipAddress, NtpClientExceptionDiscoveryFail::ErrorId::NTP_UNSYNCHRONIZED, 0,
                                    });
                                } else if (replNtpPacket.isKissOfDeath()) {
                                    discoveryErrors.emplace_back(NtpClientExceptionDiscoveryFail::Host{
                                        discoveryRequest.addr.host, discoveryRequest.addr.port, discoveryRequest.ipAddress, NtpClientExceptionDiscoveryFail::kodToErrorId(replNtpPacket.referenceIdText()), 0,
                                    });
                                } else {
                                    discoveryResults.emplace_back(std::make_shared<DiscoveryResult>(DiscoveryResult{discoveryRequest.addr, discoveryRequest.ipAddress, discoveryRequest.addrInfo, replNtpPacket}));
                                    --discoveryCount;
                                }
                            } else {
                                discoveryErrors.emplace_back(NtpClientExceptionDiscoveryFail::Host{
                                    discoveryRequest.addr.host, discoveryRequest.addr.port, discoveryRequest.ipAddress, NtpClientExceptionDiscoveryFail::ErrorId::RECV, errno,
                                });
                            }
                            discoveryRequest.socket.reset();
                            break;
                        }
                        if (abortFlag) {
                            throw AbortedByUser{};
                        }
                    }
                }
                if (!discoveryCount) {
                    break;
                }
            }
        }

        std::sort(discoveryResults.begin(), discoveryResults.end(),
                  [](const DiscoveryResultPtr& d1, const DiscoveryResultPtr& d2) {
                      return (d1->ntpPacket.destinationTime() - d1->ntpPacket.originTime()) < (d2->ntpPacket.destinationTime() - d2->ntpPacket.originTime());
                  });

        if (discoveryResults.empty()) {
            throw NtpClientExceptionDiscoveryFail(discoveryErrors);
        }

        return discoveryResults;
    }

} // namespace quasar
