#pragma once

#include <yandex_io/libs/json_utils/json_utils.h>

#include <atomic>
#include <chrono>
#include <string>
#include <vector>

namespace quasar {

    struct NtpClientApi;

    class NtpClient {
    public:
        using duration = std::chrono::steady_clock::duration;
        using time_point = std::chrono::steady_clock::time_point;
        enum class ServerTimeCredibility {
            UNDEFINED,
            APPROXIMATE, // Server time is received from the server itself and network and other delays are not taken into account
            RELIABLY,    // Server time is indicated taking into account ping and other delays and calculated at the same moment as local time
        };

        enum class SyncMode {
            DEFAULT,
            RANDOMIZE,
            DISCOVERY,

            CUSTOM
        };

        struct Addr {
            std::string host;
            uint32_t port = 123;

            explicit Addr(std::string inHost = "");
            Addr(std::string inHost, uint16_t inPort);
            bool operator<(const Addr& other) const;
            bool operator==(const Addr& other) const;
            bool operator!=(const Addr& other) const;
        };

        struct Params {
            std::chrono::milliseconds timeout{0};
            size_t minMeasuringCount{1};
            size_t sufficientMeasuringCount{1};
            size_t maxMeasuringCount{1};
            std::vector<Addr> ntpServers;

            bool operator==(const Params& other) const;
            bool operator!=(const Params& other) const;
            bool tryParseJson(const Json::Value& config) noexcept;
            void parseJson(const Json::Value& config);
            Json::Value toJson() const noexcept;
        };

        struct NtpResult {
            SyncMode syncMode;
            std::vector<Addr> resolvedIps;
            Addr addr;
            std::string ipAddress;
            std::string referenceId;
            ServerTimeCredibility serverTimeCredibility;
            duration diff;

            NtpResult();
            NtpResult(SyncMode /*inSyncMode*/, std::vector<Addr> inResolvedIps, Addr /*intAddr*/, std::string inIpaddress, std::string inReferenceId, ServerTimeCredibility /*inServerTimeCredibility*/, duration /*inDiff*/);
            time_point syncTime() const;
            static time_point systemTime();
            std::chrono::nanoseconds syncDiff() const;    // The difference between synchronized time and system
            std::chrono::nanoseconds absSyncDiff() const; // The absolute  difference between synchronized time and system
        };

        NtpClient(Params params);
        NtpClient(Params params, std::shared_ptr<const NtpClientApi> api);

        // Synchronizes the system time via NTP, in case of an error throw std::runtime_error exception
        NtpResult sync(SyncMode syncMode) const;
        NtpResult sync(SyncMode syncMode, const std::atomic<bool>& abortFlag) const;

    private:
        struct DiscoveryResult;
        using DiscoveryResultPtr = std::shared_ptr<DiscoveryResult>;
        std::vector<DiscoveryResultPtr> discoveryRequest(std::chrono::milliseconds discoveryTimeoutMs, size_t discoveryCount, const std::atomic<bool>& abortFlag) const;

    private:
        const std::shared_ptr<const NtpClientApi> api_;
        Params params_;
    };

} // namespace quasar
