#include "ntp_client_exception.h"

#include <netdb.h>

namespace quasar {
    NtpClientException::NtpClientException(std::string inMessage)
        : message(std::move(inMessage))
    {
    }

    NtpClientException::NtpClientException(std::string inMessage, std::string inHost, uint16_t inPort, std::string inIpAddress)
        : message(std::move(inMessage))
        , host(std::move(inHost))
        , port(inPort)
        , ipAddress(std::move(inIpAddress))
    {
    }

    const char* NtpClientException::what() const noexcept {
        if (whatMessage_.empty()) {
            whatMessage_.reserve(message.length() + host.length() + ipAddress.length() + 32);
            if (host.length()) {
                whatMessage_.append(host);
                whatMessage_.append(":");
                whatMessage_.append(std::to_string(port));
            }
            if (ipAddress.length() && ipAddress != host) {
                bool braket = !whatMessage_.empty();
                if (braket) {
                    whatMessage_.append(" (");
                }
                whatMessage_.append(ipAddress);
                if (braket) {
                    whatMessage_.append(")");
                }
            }
            if (whatMessage_.length()) {
                whatMessage_.append(": ");
            }
            whatMessage_.append(message);
        }
        return whatMessage_.c_str();
    }

    NtpClientExceptionWrongTime::NtpClientExceptionWrongTime(std::string inHost, uint16_t inPort, std::string inIpAddress)
        : NtpClientException("Wrong time received", std::move(inHost), inPort, std::move(inIpAddress))
    {
    }

    NtpClientExceptionTimeForgery::NtpClientExceptionTimeForgery(std::string inHost, uint16_t inPort, std::string inIpAddress)
        : NtpClientException("Original timestamp was changed", std::move(inHost), inPort, std::move(inIpAddress))
    {
    }

    NtpClientExceptionClockUnsynchronized::NtpClientExceptionClockUnsynchronized(std::string inHost, uint16_t inPort, std::string inIpAddress)
        : NtpClientException("Clock unsynchronized", std::move(inHost), inPort, std::move(inIpAddress))
    {
    }

    std::string NtpClientExceptionKissOfDeath::codeToMessage(const std::string& code)
    {
        // https://tools.ietf.org/html/rfc5905#section-7.4

        if (code == "ACST") {
            return "The association belongs to a unicast server (ACST)";
        } else if (code == "AUTH") {
            return "Server authentication failed (AUTH)";
        } else if (code == "AUTO") {
            return "Autokey sequence failed (AUTO)";
        } else if (code == "BCST") {
            return "The association belongs to a broadcast server (BCST)";
        } else if (code == "CRYP") {
            return "Cryptographic authentication or identification failed (CRYP)";
        } else if (code == "DENY") {
            return "Access denied by remote server (DENY)";
        } else if (code == "DROP") {
            return "Lost peer in symmetric mode (DROP)";
        } else if (code == "RSTR") {
            return "Access denied due to local policy (RSTR)";
        } else if (code == "INIT") {
            return "The association has not yet synchronized for the first time (INIT)";
        } else if (code == "MCST") {
            return "The association belongs to a dynamically discovered server (MCST)";
        } else if (code == "NKEY") {
            return "No key found. Either the key was never installed or is not trusted (NKEY)";
        } else if (code == "NTSN") {
            return "Network Time Security (NTS) negative-acknowledgment (NAK) (NTSN)";
        } else if (code == "RATE") {
            return "Rate exceeded. The server has temporarily denied access because the client exceeded the rate threshold (RATE)";
        } else if (code == "RMOT") {
            return "Alteration of association from a remote host running ntpdc (RMOT)";
        } else if (code == "STEP") {
            return "A step change in system time has occurred, but the association has not yet resynchronized (STEP)";
        }

        return "Server return Kiss-Of-Death code " + code;
    }

    NtpClientExceptionKissOfDeath::NtpClientExceptionKissOfDeath(const std::string& inCode, std::string inHost, uint16_t inPort, std::string inIpAddress)
        : NtpClientException(codeToMessage(inCode), std::move(inHost), inPort, std::move(inIpAddress))
    {
    }

    NtpClientExceptionDiscoveryFail::ErrorId NtpClientExceptionDiscoveryFail::kodToErrorId(const std::string& code)
    {
        // https://tools.ietf.org/html/rfc5905#section-7.4

        if (code == "ACST") {
            return ErrorId::KOD_ACST;
        } else if (code == "AUTH") {
            return ErrorId::KOD_AUTH;
        } else if (code == "AUTO") {
            return ErrorId::KOD_AUTO;
        } else if (code == "BCST") {
            return ErrorId::KOD_BCST;
        } else if (code == "CRYP") {
            return ErrorId::KOD_CRYP;
        } else if (code == "DENY") {
            return ErrorId::KOD_DENY;
        } else if (code == "DROP") {
            return ErrorId::KOD_DROP;
        } else if (code == "RSTR") {
            return ErrorId::KOD_RSTR;
        } else if (code == "INIT") {
            return ErrorId::KOD_INIT;
        } else if (code == "MCST") {
            return ErrorId::KOD_MCST;
        } else if (code == "NKEY") {
            return ErrorId::KOD_NKEY;
        } else if (code == "NTSN") {
            return ErrorId::KOD_NTSN;
        } else if (code == "RATE") {
            return ErrorId::KOD_RATE;
        } else if (code == "RMOT") {
            return ErrorId::KOD_RMOT;
        } else if (code == "STEP") {
            return ErrorId::KOD_STEP;
        }
        return ErrorId::KOD_OTHER;
    }

    NtpClientExceptionDiscoveryFail::NtpClientExceptionDiscoveryFail(std::vector<Host> hosts)
        : NtpClientException("Fail to discovery any ntp servers: [")
    {
        std::sort(hosts.begin(), hosts.end(),
                  [](const Host& h1, const Host& h2) {
                      if (h1.host != h2.host) {
                          return h1.host < h2.host;
                      } else if (h1.port != h2.port) {
                          return h1.port < h2.port;
                      } else if (h1.ipAddress != h2.ipAddress) {
                          return h1.ipAddress < h2.ipAddress;
                      } else if (h1.errorId != h2.errorId) {
                          return h1.errorId < h2.errorId;
                      }
                      return h1.errNo < h2.errNo;
                  });
        message.reserve(message.size() * hosts.size() * 64);
        bool first = true;
        for (const auto& host : hosts) {
            if (!first) {
                message.append(", ");
            }
            message.append(host.host);
            if (host.port) {
                message.append(":");
                message.append(std::to_string(host.port));
            }
            if (!host.ipAddress.empty() && host.ipAddress != host.host) {
                message.append(" (");
                message.append(host.ipAddress);
                message.append(")");
            }
            switch (host.errorId) {
                case ErrorId::GETADDRINFO:
                    message.append(" getaddrinfo");
                    break;
                case ErrorId::SEND:
                    message.append(" send");
                    break;
                case ErrorId::TIMEOUT:
                    message.append(" timeout");
                    break;
                case ErrorId::RECV:
                    message.append(" recv");
                    break;
                case ErrorId::NTP_TIME_FORGERY:
                    message.append(" timeforgery");
                    break;
                case ErrorId::NTP_UNSYNCHRONIZED:
                    message.append(" unsynchronized");
                    break;
                case ErrorId::KOD_ACST:
                    message.append(" KOD_ACST");
                    break;
                case ErrorId::KOD_AUTH:
                    message.append(" KOD_AUTH");
                    break;
                case ErrorId::KOD_AUTO:
                    message.append(" KOD_AUTO");
                    break;
                case ErrorId::KOD_BCST:
                    message.append(" KOD_BCST");
                    break;
                case ErrorId::KOD_CRYP:
                    message.append(" KOD_CRYP");
                    break;
                case ErrorId::KOD_DENY:
                    message.append(" KOD_DENY");
                    break;
                case ErrorId::KOD_DROP:
                    message.append(" KOD_DROP");
                    break;
                case ErrorId::KOD_RSTR:
                    message.append(" KOD_RSTR");
                    break;
                case ErrorId::KOD_INIT:
                    message.append(" KOD_INIT");
                    break;
                case ErrorId::KOD_MCST:
                    message.append(" KOD_MCST");
                    break;
                case ErrorId::KOD_NKEY:
                    message.append(" KOD_NKEY");
                    break;
                case ErrorId::KOD_NTSN:
                    message.append(" KOD_NTSN");
                    break;
                case ErrorId::KOD_RATE:
                    message.append(" KOD_RATE");
                    break;
                case ErrorId::KOD_RMOT:
                    message.append(" KOD_RMOT");
                    break;
                case ErrorId::KOD_STEP:
                    message.append(" KOD_STEP");
                    break;
                case ErrorId::KOD_OTHER:
                    message.append(" KOD_OTHER");
                    break;
                case ErrorId::OTHER:
                    message.append(" other");
                    break;
            }
            if (host.errNo) {
                message.append(" errno=");
                message.append(std::to_string(host.errNo));
            }
            if (host.errOther) {
                message.append(" errEx=");
                if (host.errorId == ErrorId::GETADDRINFO) {
                    switch (host.errOther)
                    {
                        case EAI_ADDRFAMILY:
                            message.append("EAI_ADDRFAMILY");
                            break;
                        case EAI_AGAIN:
                            message.append("EAI_AGAIN");
                            break;
                        case EAI_BADFLAGS:
                            message.append("EAI_BADFLAGS");
                            break;
                        case EAI_FAIL:
                            message.append("EAI_FAIL");
                            break;
                        case EAI_FAMILY:
                            message.append("EAI_FAMILY");
                            break;
                        case EAI_MEMORY:
                            message.append("EAI_MEMORY");
                            break;
                        case EAI_NODATA:
                            message.append("EAI_NODATA");
                            break;
                        case EAI_NONAME:
                            message.append("EAI_NONAME");
                            break;
                        case EAI_SERVICE:
                            message.append("EAI_SERVICE");
                            break;
                        case EAI_SOCKTYPE:
                            message.append("EAI_SOCKTYPE");
                            break;
                        case EAI_SYSTEM:
                            message.append("EAI_SYSTEM");
                            break;
                        default:
                            message.append(std::to_string(host.errOther));
                            break;
                    }
                } else {
                    message.append(std::to_string(host.errOther));
                }
            }
            first = false;
        }
        message.append("]");
    }

} // namespace quasar
